
/***************************************************************************

                                 ChessV

                   COPYRIGHT (C) 2005 BY GREGORY STRONG

This file is part of ChessV.  ChessV is free software; you can redistribute
it and/or modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of the License, 
or (at your option) any later version.

ChessV is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
more details; the file 'COPYING' contains the License text, but if for
some reason you need a copy, please write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/


#ifndef FILE__MOVEMENT_CAPABILITY_H
#define FILE__MOVEMENT_CAPABILITY_H


#include "Player.h"
#include "Zone.h"


#define MC_MAX		32


enum MovementType
{
	Normal = 1,
	//	like normal chess pieces

	MoveOnly = 2,
	//	like the pawn's forward move

	CaptureOnly = 3,
	//	like the pawn's diagonal capture

	IguiCaptureOnly = 4,
	//	may not move to this square, but may
	//	capture a piece on it without moving at all

	AreaMover = 5
	//	makes successive single steps
};


class MovementCapability
{
  protected:
	//	data members
	int directionNumber;				//	direction number of movement
    int fileDiff;						//	change (positive or negative) of file
	int rankDiff;						//	change (positive or negative) of rank
	int maxSteps;						//	maximum distance in this direction
	int minSteps;						//	minimum distance in this direction (lame mover)

	MovementType type;
	bool withdrawingCaptures;
	bool advancingCaptures;

	//	conditions
	Zone restrictionZone;


  public:
	//	construction
	MovementCapability();

	MovementCapability
		( int fileChange, 
		  int rankChange,
		  int steps );

	MovementCapability
		( int fileChange, 
		  int rankChange,
		  int steps,
		  MovementType moveType );

	MovementCapability
		( int fileChange, 
		  int rankChange,
		  int steps,
		  MovementType moveType, 
		  int minSquareNumber, 
		  int maxSquareNumber, 
		  bool withdrawCapture = false, 
		  bool advanceCapture = false,
		  int minimumSteps = 1 );

	MovementCapability
		( MovementCapability const &other );


	//	assignment
	MovementCapability &operator=
		( MovementCapability const &other );


	//	attributes
	int GetFileDifference( const Player &player ) const;

	int GetFileDifference( int playerNumber ) const;

	int GetRankDifference( const Player &player ) const;

	int GetRankDifference( int playerNumber ) const;

	int GetMaxSteps() const
	{	return maxSteps == 0 ? 128 : maxSteps;  }

	int GetMinSteps() const
	{	return minSteps;  }

	bool IsInZone( int squareNumber )
	{	return restrictionZone.IsInZone( squareNumber );   }

	Zone &GetRestrictionZone()
	{	return restrictionZone;   }

	MovementType GetType() const
	{	return type;   }

	bool CanMove() const
	{	return type == Normal || type == MoveOnly;   }

	bool CanCapture() const
	{	return type == Normal || type == CaptureOnly;   }

	bool CanIguiCapture() const
	{	return type == IguiCaptureOnly;   }

	bool CapturesByWithdrawing() const
	{	return withdrawingCaptures;   }

	bool CapturesByAdvancing() const
	{	return advancingCaptures;   }

	int GetDirectionNumber()
	{	return directionNumber;   }

	void SetDirectionNumber( int number )
	{	directionNumber = number;   }
};


#endif
