
/***************************************************************************

                                 ChessV

                   COPYRIGHT (C) 2005 BY GREGORY STRONG

This file is part of ChessV.  ChessV is free software; you can redistribute
it and/or modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of the License, 
or (at your option) any later version.

ChessV is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
more details; the file 'COPYING' contains the License text, but if for
some reason you need a copy, please write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/


#include "StdAfx.h"
#include "../../ChessV.h"
#include "../../PieceType.h"
#include "../../Piece.h"
#include "../../Direction.h"
#include "../../Rand.h"
#include "../../GameParameters.h"
#include "ArchchessGame.h"


#include "../10x10/Decimal_Data.h"
//	these include file contains data matricies for square bonuses,
//	outpost bonuses, etc.  it also contains some macros used here.


DecimalDecurion DecimalDecurion::decimalDecurion;
DecimalElephant DecimalElephant::decimalElephant;
DecimalFerz DecimalFerz::decimalFerz;
DecimalWazir DecimalWazir::decimalWazir;
DecimalGeneral DecimalGeneral::decimalGeneral;



Game *CreateArchchessGame( Board &brd, Player &plr0, Player &plr1 )
{
	return new ArchchessGame( brd, plr0, plr1 );
}


ArchchessGame::ArchchessGame( Board &board, Player &whitePlayer, Player &blackPlayer ):
	Game(board, whitePlayer, blackPlayer)
{
	// ***************************************** //
	// ***                                   *** //
	// ***  set rules and evaluation values  *** //
	// ***                                   *** //
	// ***************************************** //

	//	implement 50-move draw rule (100 half-moves)
	autoDrawPeriod = 100;

	//	turn on use of pawn structure evaluation
	usePawnStructureEvaluation = true;

	//	turn off null-move after 30 pieces have been captured
	endgameCaptureThreshold = 30;

	//	razoring and futility pruning margins
	razorMargin = 2500;
	futilityMargin = 1250;
	extendedFutilityMargin = 2200;

	// *** PHASES *** //

	//	we have to set up the phases we wish to have in this game;
	//	we will keep it simple and have 3: opening, midgame, and endgame
	nPhases = 3;

	//	settings for phase 0 (opening)
	phases[0].SetNumber( 0 );
	phases[0].SetMobilityFactor( 1 );
	phases[0].SetPawnDeficiencyFactor( 8 );
	phases[0].SetSquareValuesFactor( 1 );
	phases[0].SetKingSafetyFactor( 0 );
	phases[0].SetTropismFactor( 6 );

	//	settings for phase 1 (midgame)
	phases[1].SetNumber( 1 );
	phases[1].SetMobilityFactor( 1 );
	phases[1].SetPawnDeficiencyFactor( 10 );
	phases[1].SetSquareValuesFactor( 1 );
	phases[1].SetKingSafetyFactor( 0 );
	phases[1].SetTropismFactor( 10 );
	
	//	settings for phase 2 (endgame)
	phases[2].SetNumber( 2 );
	phases[2].SetMobilityFactor( 0 );
	phases[2].SetPawnDeficiencyFactor( 16 );
	phases[2].SetSquareValuesFactor( 1 );
	phases[2].SetKingSafetyFactor( 0 );
	phases[2].SetTropismFactor( 10 );

	// *** OUTPOSTS *** //

	//	we wish to be able to give bonuses to pieces that are 'posted'-
	//	that is, which are centrally located and cannot be reached by
	//	enemy pawns (because adjacent pawns are gone, or have advanced
	//	beyond the piece).  we also wish to be able to give different
	//	bonuses for different types of minor pieces (a posted Knight is
	//	worth more than a posted Bishop.)

	//	we will use the user variable 'gameInt1' in the PieceType class
	//	to store the 'outpost factor' for a given piece.  this value will
	//	be multiplied by the square-values stored in the 'outpost' array.
	DecimalKnight::decimalKnight.gameInt1 = 10;
	DecimalBishop::decimalBishop.gameInt1 = 6;
	DecimalDecurion::decimalDecurion.gameInt1 = 10;
	DecimalFerz::decimalFerz.gameInt1 = 2;


	// *** INITIALIZATION *** //

	//	initialize hashes
	enPassantHashMap = new HashMap( board.GetNumberOfSquares() );

	//	basic initialization
	board.Initialize( this, BITBOARD_128 );
}

void ArchchessGame::AddPlayerPieceTypes()
{
	//	add the piece types that each player may have
	board.AddPlayerPieceTypeBothPlayers( DecimalRook::decimalRook );
	board.AddPlayerPieceTypeBothPlayers( DecimalBishop::decimalBishop );
	board.AddPlayerPieceTypeBothPlayers( DecimalElephant::decimalElephant );
	board.AddPlayerPieceTypeBothPlayers( DecimalKnight::decimalKnight );
	board.AddPlayerPieceTypeBothPlayers( DecimalFerz::decimalFerz );
	board.AddPlayerPieceTypeBothPlayers( DecimalKing::decimalKing );
	board.AddPlayerPieceTypeBothPlayers( DecimalPawn::decimalPawn );
	board.AddPlayerPieceTypeBothPlayers( DecimalQueen::decimalQueen );
	board.AddPlayerPieceTypeBothPlayers( DecimalDecurion::decimalDecurion );
}

void ArchchessGame::InitializeBoard()
{
	//	add the piece types that each player may have
	AddPlayerPieceTypes();

	//	clear out all piece references
	for( int player = 0; player < 2; player++ )
	{
		queens[player] = NULL;
		for( int x = 0; x < 6; x++ )
		{
			if( x < 2 )
				rooks[player][x] = NULL;
			minorPieces[player][x] = NULL;
		}
	}

	//	change the names of the piece types appropriately
	DecimalFerz::decimalFerz.SetFullName( "Decurion" );
	DecimalFerz::decimalFerz.SetNotation( "D" );
	DecimalDecurion::decimalDecurion.SetFullName( "Centurion" );
	DecimalDecurion::decimalDecurion.SetNotation( "C" );
	DecimalBishop::decimalBishop.SetBitmapFileName( PIECE_SET_OLD_WORLD, 0, "images\\old_world\\WBishop.bmp" );
	DecimalBishop::decimalBishop.SetBitmapFileName( PIECE_SET_OLD_WORLD, 1, "images\\old_world\\BBishop.bmp" );
	DecimalBishop::decimalBishop.SetFullName( "Bishop" );
	DecimalBishop::decimalBishop.SetNotation( "B" );

	//	pawns promote only to queens
	DecimalPawn::decimalPawn.SetPromotionType( PromoteToSpecificType );
	DecimalPawn::decimalPawn.SetTypeToPromoteTo( DecimalFerz::decimalFerz );

	//	place pieces
	char *FEN = "rnbckqdbnr/pppppppppp/10/10/10/10/10/10/PPPPPPPPPP/RNBCKQDBNR";
	if( LookupStringParameter( "array", FEN ) )
		useOpeningBook = false;
	board.PlacePiecesByFEN( FEN );

	//	opening book
	if( useOpeningBook )
	{
		Book *pBook = new Book( &board );
		board.SetOpeningBook( pBook, "openings\\Archchess.txt" );
	}

	board.PostInitialize();
}

Piece *ArchchessGame::AddPiece
	( PieceType &pieceType,
	  int nPlayer,
	  int nRank,
	  int nFile )
{
	Piece *newPiece = Game::AddPiece( pieceType, nPlayer, nRank, nFile );

	if( pieceType == DecimalRook::decimalRook )
	{
		AddPieceToSet( newPiece, rooks, 2 );
	}
	else if( pieceType == DecimalKnight::decimalKnight ||
		     pieceType == DecimalBishop::decimalBishop || 
		     pieceType == DecimalDecurion::decimalDecurion || 
		     pieceType == DecimalFerz::decimalFerz )
	{
		AddPieceToSet( newPiece, minorPieces, 6 );
	}
	else if( newPiece->GetType().IsPawn() )
	{
		if( newPiece->GetPlayerNumber() == 0 )
			newPiece->GetPromotionZone().AddToZone( 90, 99 );
		else
			newPiece->GetPromotionZone().AddToZone( 0, 9 );
	}
	return newPiece;
}

void ArchchessGame::DeletePiece
	( Piece *piece )
{
	if( queens[0] == piece )
		queens[0] = NULL;
	if( queens[1] == piece )
		queens[1] = NULL;
	RemovePieceFromSet( piece, minorPieces, 6 );
	RemovePieceFromSet( piece, rooks, 2 );
}

PieceType **ArchchessGame::GetPieceTypesRequired
	( int &nPieceTypes )
{
	nPieceTypes = board.GetPieceTypeCount();
	return( board.GetPieceTypes() );
}

void ArchchessGame::AddSpecialMoves
	( int currentPlayer, 
	  MovementList &stack, 
	  GameRec &gameRecord,
	  bool quiescentSearch )
{
	//	check for en-passant
	Game::AddEnPassantMoves( stack, gameRecord, quiescentSearch );

	//	see if we can castle
	if( board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() )
		{
			//	king side
			if( rooks[0][1] != NULL &&
				!rooks[0][1]->IsCaptured() &&
				!rooks[0][1]->HasMoved() )
			{
				if( board.GetSquareContents( 5 ) == NULL &&
					board.GetSquareContents( 6 ) == NULL &&
					board.GetSquareContents( 7 ) == NULL &&
					board.GetSquareContents( 8 ) == NULL )
				{
					//	squares are vacant; make sure they are not attacked
					if( !IsSquareAttacked( 5, 1 ) &&
						!IsSquareAttacked( 6, 1 ) && 
						!IsSquareAttacked( 7, 1 ) )
					{
						stack.BeginMoveAdd( Castling, 4, 7 );
						stack.AddPickUp( 4 );
						stack.AddPickUp( 9 );
						stack.AddDrop( king, 7 );
						stack.AddDrop( rooks[0][1], 6 );
						stack.EndMoveAdd( 1000 );
					}
				}
			}
			//	queen side
			if( rooks[0][0] != NULL &&
				!rooks[0][0]->IsCaptured() &&
				!rooks[0][0]->HasMoved() )
			{
				if( board.GetSquareContents( 3 ) == NULL &&
					board.GetSquareContents( 2 ) == NULL && 
					board.GetSquareContents( 1 ) == NULL )
				{
					//	square are vacant; make sure they are not attacked
					if( !IsSquareAttacked( 3, 1 ) && 
						!IsSquareAttacked( 2, 1 ) && 
						!IsSquareAttacked( 1, 1 ) )
					{
						stack.BeginMoveAdd( Castling, 4, 1 );
						stack.AddPickUp( 4 );
						stack.AddPickUp( 0 );
						stack.AddDrop( king, 1 );
						stack.AddDrop( rooks[0][0], 2 );
						stack.EndMoveAdd( 1000 );
					}
				}
			}
			//	king's leap
			if( board.GetSquareContents( 2 ) == NULL &&
				!IsSquareAttacked( 2, 1 ) )
			{
				stack.BeginMoveAdd( UserMove1, 4, 2 );
				stack.AddPickUp( 4 );
				stack.AddDrop( king, 2 );
				stack.EndMoveAdd( 0 );
			}
			if( board.GetSquareContents( 6 ) == NULL &&
				!IsSquareAttacked( 6, 1 ) )
			{
				stack.BeginMoveAdd( UserMove1, 4, 6 );
				stack.AddPickUp( 4 );
				stack.AddDrop( king, 6 );
				stack.EndMoveAdd( 0 );
			}
			if( board.GetSquareContents( 24 ) == NULL && 
				!IsSquareAttacked( 24, 1 ) )
			{
				stack.BeginMoveAdd( UserMove1, 4, 24 );
				stack.AddPickUp( 4 );
				stack.AddDrop( king, 24 );
				stack.EndMoveAdd( 0 );
			}
		}
	}
	else if( board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() )
		{
			//	king side
			if( rooks[1][1] != NULL &&
				!rooks[1][1]->IsCaptured() &&
				!rooks[1][1]->HasMoved() )
			{
				if( board.GetSquareContents( 95 ) == NULL &&
					board.GetSquareContents( 96 ) == NULL && 
					board.GetSquareContents( 97 ) == NULL &&
					board.GetSquareContents( 98 ) == NULL )
				{
					//	square are vacant; make sure they are not attacked
					if( !IsSquareAttacked( 95, 0 ) && 
						!IsSquareAttacked( 96, 0 ) && 
						!IsSquareAttacked( 97, 0 ) )
					{
						stack.BeginMoveAdd( Castling, 94, 97 );
						stack.AddPickUp( 94 );
						stack.AddPickUp( 99 );
						stack.AddDrop( king, 97 );
						stack.AddDrop( rooks[1][1], 96 );
						stack.EndMoveAdd( 1000 );
					}
				}
			}
			//	queen side
			if( rooks[1][0] != NULL &&
				!rooks[1][0]->IsCaptured() &&
				!rooks[1][0]->HasMoved() )
			{
				if( board.GetSquareContents( 93 ) == NULL &&
					board.GetSquareContents( 92 ) == NULL && 
					board.GetSquareContents( 91 ) == NULL )
				{
					//	squares are vacant; make sure they are not attacked
					if( !IsSquareAttacked( 93, 0 ) && 
						!IsSquareAttacked( 92, 0 ) && 
						!IsSquareAttacked( 91, 0 ) )
					{
						stack.BeginMoveAdd( Castling, 94, 91 );
						stack.AddPickUp( 94 );
						stack.AddPickUp( 90 );
						stack.AddDrop( king, 91 );
						stack.AddDrop( rooks[1][0], 92 );
						stack.EndMoveAdd( 1000 );
					}
				}
			}
			//	king's leap
			if( board.GetSquareContents( 92 ) == NULL )
			{
				stack.BeginMoveAdd( UserMove1, 94, 92 );
				stack.AddPickUp( 94 );
				stack.AddDrop( king, 92 );
				stack.EndMoveAdd( 0 );
			}
			if( board.GetSquareContents( 96 ) == NULL )
			{
				stack.BeginMoveAdd( UserMove1, 94, 96 );
				stack.AddPickUp( 94 );
				stack.AddDrop( king, 96 );
				stack.EndMoveAdd( 0 );
			}
			if( board.GetSquareContents( 74 ) == NULL )
			{
				stack.BeginMoveAdd( UserMove1, 94, 74 );
				stack.AddPickUp( 94 );
				stack.AddDrop( king, 74 );
				stack.EndMoveAdd( 0 );
			}
		}
	}
}

bool ArchchessGame::MoveBeingMade
	( MoveInfo &moveInfo,
	  GameRec &gameRecord )
{
	//	special 2-move pawn rule restriction;  a 
	//	a 2-move pawn move is illegal when it may 
	//	be taken en passant, and also blocks check,
	//	so we render those moves illegal here
	if( board.FastCheckTest() && 
		gameRecord.pieceMoved->IsPawn() &&
		(gameRecord.pieceMoved->GetFlags() & (FLAGS_HAS_MOVED | FLAGS_HAS_MOVED_TWICE)) == FLAGS_HAS_MOVED && 
		((gameRecord.pieceMoved->GetPlayerNumber() == 0 && gameRecord.pieceMoved->GetSquareNumber() >= 30 && 
		  gameRecord.pieceMoved->GetSquareNumber() < 40) ||
	     (gameRecord.pieceMoved->GetPlayerNumber() == 1 && gameRecord.pieceMoved->GetSquareNumber() >= 60 && 
		  gameRecord.pieceMoved->GetSquareNumber() < 70)) )
	{
		//	possible illegal move; see if there's a pawn 
		//	that can capture this piece en passant
		int sq = board.GetMovementMatrix( DIRECTION_E )[gameRecord.pieceMoved->GetSquareNumber()];
		if( sq != -1 && board.GetSquareContents( sq ) != NULL )
			if( board.GetSquareContents( sq )->IsPawn() && 
				board.GetSquareContents( sq )->GetPlayerNumber() != gameRecord.pieceMoved->GetPlayerNumber() )
				//	illegal move
				return false;
		sq = board.GetMovementMatrix( DIRECTION_W )[gameRecord.pieceMoved->GetSquareNumber()];
		if( sq != -1 && board.GetSquareContents( sq ) != NULL )
			if( board.GetSquareContents( sq )->IsPawn() && 
				board.GetSquareContents( sq )->GetPlayerNumber() != gameRecord.pieceMoved->GetPlayerNumber() )
				//	illegal move
				return false;
	}

	// *** UPDATE gameInt1 *** //

	//	gameInt1 is a 'user' field in the game records that a game can use
	//	for any purpose.  we will use it to track the forward progress of 
	//	the game, so we can determine which Phase we are in ('opening', 
	//	'midgame', or 'endgame' in our case)

	if( gameRecord.gameInt1 < 20 )
	{
		//	we are in the opening until gameInt1 reaches 20;
		//	in the opening, we increment gameInt1 for the following moves:
		//		- any capture
		//		- the first move of any piece
		//		- any pawn push
		//		- king's leap adds 2
		//		- castling adds 3
		if( gameRecord.materialCaptured > 0 ||
			(gameRecord.pieceMoved->GetFlags() & (FLAGS_HAS_MOVED | FLAGS_HAS_MOVED_TWICE)) == FLAGS_HAS_MOVED ||
			gameRecord.pieceMoved->IsPawn() )
			gameRecord.gameInt1++;
		else if( gameRecord.moveType == Castling )
			gameRecord.gameInt1 += 3;
		else if( gameRecord.moveType == UserMove1 )
			gameRecord.gameInt1 += 2;
	}

	return true;
}

Phase &ArchchessGame::AdjustEvaluation
	( int &eval,
	  PawnHash *pPawnHash )
{
	Phase *currentPhase = phases;

	if( board.GetCurrentGameRecord().gameInt1 < 20 )
	{
		// *** The OPENING *** //
		//	don't block the king or queen's pawns in the opening!!!
		if( board.GetSquareContents( 24 ) != NULL &&
			board.GetSquareContents( 14 ) != NULL &&
			!board.GetSquareContents( 14 )->HasMoved() )
			eval -= 160;
		if( board.GetSquareContents( 25 ) != NULL &&
			board.GetSquareContents( 15 ) != NULL &&
			!board.GetSquareContents( 15 )->HasMoved() )
			eval -= 160;
		if( board.GetSquareContents( 74 ) != NULL &&
			board.GetSquareContents( 84 ) != NULL &&
			!board.GetSquareContents( 84 )->HasMoved() )
			eval += 160;
		if( board.GetSquareContents( 75 ) != NULL &&
			board.GetSquareContents( 85 ) != NULL &&
			!board.GetSquareContents( 85 )->HasMoved() )
			eval += 160;
	}
	else
		currentPhase = phases + 1;


	// *** OUTPOSTS *** //

	//	outposts are minor pieces in central squares that cannot be driven off by enemy pawns.
	//	the basic output bonus is the value of the piece's square in the 'outpost' array, times
	//	the value of the piece type's 'outpost factor' (stored in 'gameInt1'.)

	//	a posted piece that is protected by one of its own pawns is extra good, so
	//	it gets 150% of the above bonus.  a posted piece protected by two of its
	//	own pawns is really, really good, because if the posted piece is captured,
	//	the player will have a protected, passed pawn.  a posted piece that is thus
	//	protected by 2 pawns gets double the standard output bonus.

	//	loop through all minor pieces
	for( int x = 0; x < 6 && pPawnHash != NULL; x++ )
	{
		if( minorPieces[0][x] != NULL &&
			!minorPieces[0][x]->IsCaptured() &&
			outpost_10x10[minorPieces[0][x]->GetSquareNumber()] > 0 &&
			minorPieces[0][x]->GetType().gameInt1 > 0 &&
			((int) (pPawnHash->nPawnsPerFile[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]-1]) == 0 ||
			 (int) (pPawnHash->backPawnRank[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]-1]) <= board.ranks[0][minorPieces[0][x]->GetSquareNumber()]) &&
			((int) (pPawnHash->nPawnsPerFile[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]+1]) == 0 ||
			 (int) (pPawnHash->backPawnRank[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]+1]) <= board.ranks[0][minorPieces[0][x]->GetSquareNumber()]) )
		{
			//	basic bonus for player 0's posted piece
			eval += outpost_10x10[minorPieces[0][x]->GetSquareNumber()] *
			        minorPieces[0][x]->GetType().gameInt1;
			int pawnSquare1 = board.GetMovementMatrix( DIRECTION_SE )[minorPieces[0][x]->GetSquareNumber()];
			int pawnSquare2 = board.GetMovementMatrix( DIRECTION_SW )[minorPieces[0][x]->GetSquareNumber()];
			if( board.GetSquareContents( pawnSquare1 ) != NULL &&
				board.GetSquareContents( pawnSquare1 )->GetType().IsPawn() )
				//	additional 50% bonus for this pawn defender
				eval += (outpost_10x10[minorPieces[0][x]->GetSquareNumber()] *
				         minorPieces[0][x]->GetType().gameInt1) >> 1;
			if( board.GetSquareContents( pawnSquare2 ) != NULL &&
				board.GetSquareContents( pawnSquare2 )->GetType().IsPawn() )
				//	additional 50% bonus for this pawn defender
				eval += (outpost_10x10[minorPieces[0][x]->GetSquareNumber()] *
				         minorPieces[0][x]->GetType().gameInt1) >> 1;
		}
		if( minorPieces[1][x] != NULL &&
			!minorPieces[1][x]->IsCaptured() &&
			outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] > 0 &&
			minorPieces[1][x]->GetType().gameInt1 > 0 &&
			((int) (pPawnHash->nPawnsPerFile[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]-1]) == 0 ||
			 (int) (pPawnHash->backPawnRank[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]-1]) >= board.ranks[0][minorPieces[1][x]->GetSquareNumber()]) &&
			((int) (pPawnHash->nPawnsPerFile[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]+1]) == 0 ||
			 (int) (pPawnHash->backPawnRank[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]+1]) >= board.ranks[0][minorPieces[1][x]->GetSquareNumber()]) )
		{
			//	basic bonus for player 1's posted piece
			eval -= outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] *
			        minorPieces[1][x]->GetType().gameInt1;
			int pawnSquare1 = board.GetMovementMatrix( DIRECTION_NE )[minorPieces[1][x]->GetSquareNumber()];
			int pawnSquare2 = board.GetMovementMatrix( DIRECTION_NW )[minorPieces[1][x]->GetSquareNumber()];
			if( board.GetSquareContents( pawnSquare1 ) != NULL &&
				board.GetSquareContents( pawnSquare1 )->GetType().IsPawn() )
				//	additional 50% bonus for this pawn defender
				eval -= (outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] *
				         minorPieces[1][x]->GetType().gameInt1) >> 1;
			if( board.GetSquareContents( pawnSquare2 ) != NULL &&
				board.GetSquareContents( pawnSquare2 )->GetType().IsPawn() )
				//	additional 50% bonus for this pawn defender
				eval -= (outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] *
				         minorPieces[1][x]->GetType().gameInt1) >> 1;
		}
	}

	// *** ROOK ON OPEN and SEMI-OPEN FILE BONUS *** //

	if( pPawnHash != NULL )
	{
		if( rooks[0][0] != NULL &&
			!rooks[0][0]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[0][board.files[0][rooks[0][0]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval += 50;
			if( pPawnHash->nPawnsPerFile[1][board.files[0][rooks[0][0]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval += 20;
		}
		if( rooks[0][1] != NULL &&
			!rooks[0][1]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[0][board.files[0][rooks[0][1]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval += 50;
			if( pPawnHash->nPawnsPerFile[1][board.files[0][rooks[0][1]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval += 20;
		}
	}
	if( pPawnHash != NULL )
	{
		if( rooks[1][0] != NULL &&
			!rooks[1][0]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[1][board.files[0][rooks[1][0]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval -= 50;
			if( pPawnHash->nPawnsPerFile[0][board.files[0][rooks[1][0]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval -= 20;
		}
		if( rooks[1][1] != NULL &&
			!rooks[1][1]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[1][board.files[0][rooks[1][1]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval -= 50;
			if( pPawnHash->nPawnsPerFile[0][board.files[0][rooks[1][1]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval -= 20;
		}
	}
	return *currentPhase;
}

int ArchchessGame::EnumeratePromotions
	( Piece *piece, 
	  int fromSquare,
	  int toSquare,
	  PieceType **promotions )
{
	int count = 0;
	return count;
}

word32 ArchchessGame::AdjustPrimaryHash
	( word32 primaryHash )
{
	return primaryHash ^ GetEnPassantHash( board.GetCurrentGameRecord(), enPassantHashMap );
}

void ArchchessGame::DefaultSettings()
{
	squareColor1 = RGB(255, 255, 220);
	squareColor2 = RGB(150, 175, 161);
	pieceColor1 = RGB(255, 255, 255);
	pieceColor2 = RGB(0, 0, 0);
	borderColor = RGB(108, 0, 0);
	boardDisplayType = BOARD_IS_CHECKERED;
	selectedPieceSet = PIECE_SET_OLD_WORLD;
}
