
/***************************************************************************

                                 ChessV

                   COPYRIGHT (C) 2005 BY GREGORY STRONG

This file is part of ChessV.  ChessV is free software; you can redistribute
it and/or modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of the License, 
or (at your option) any later version.

ChessV is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
more details; the file 'COPYING' contains the License text, but if for
some reason you need a copy, please write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/


#include "StdAfx.h"
#include "../../ChessV.h"
#include "../../PieceType.h"
#include "../../Piece.h"
#include "../../Direction.h"
#include "../../Rand.h"
#include "../../GameParameters.h"
#include "GrandChessGame.h"
#include "Decimal_Types.h"

#include "Decimal_Data.h"
//	this include file contains data matricies for square bonuses,
//	outpost_10x10 bonuses, etc.  it also contains some macros used here.


// *** DEBUG MEMORY ALLOCATION *** //
#ifdef _DEBUG
#define new SAFE_NEW
#endif


Game *CreateGrandChessGame( Board &brd, Player &plr0, Player &plr1 )
{
	return new GrandChessGame( brd, plr0, plr1 );
}

GrandChessGame::GrandChessGame( Board &board, Player &whitePlayer, Player &blackPlayer ):
	Game(board, whitePlayer, blackPlayer)
{
	// ***************************************** //
	// ***                                   *** //
	// ***  set rules and evaluation values  *** //
	// ***                                   *** //
	// ***************************************** //

	//	implement 50-move draw rule (100 half-moves)
	autoDrawPeriod = 100;

	//	turn on use of pawn structure evaluation
	usePawnStructureEvaluation = true;

	//	turn off null-move after 24 pieces have been captured
	endgameCaptureThreshold = 24;

	//	razoring and futility pruning margins
	razorMargin = 9500;
	futilityMargin = 3200;
	extendedFutilityMargin = 6500;

	//	change names of pieces
	DecimalArchbishop::decimalArchbishop.SetFullName( "Cardinal" );
	DecimalArchbishop::decimalArchbishop.SetNotation( "C" );
	DecimalChancellor::decimalChancellor.SetFullName( "Marshall" );
	DecimalChancellor::decimalChancellor.SetNotation( "M" );

	DecimalRook::decimalRook.SetSquareBonuses( 0, gcOpeningRookSquareBonuses_10x10 );
	DecimalBishop::decimalBishop.SetSquareBonuses( 0, gcOpeningBishopSquareBonuses_10x10 );


	// *** PHASES *** //

	//	we have to set up the phases we wish to have in this game;
	//	we will keep it simple and have 3: opening, midgame, and endgame
	nPhases = 3;

	//	settings for phase 0 (opening)
	phases[0].SetNumber( 0 );
	phases[0].SetMobilityFactor( 0 );
	phases[0].SetPawnDeficiencyFactor( 12 );
	phases[0].SetSquareValuesFactor( 1 );
	phases[0].SetKingSafetyFactor( 0 );
	phases[0].SetTropismFactor( 6 );

	//	settings for phase 1 (midgame)
	phases[1].SetNumber( 1 );
	phases[1].SetMobilityFactor( 0 );
	phases[1].SetPawnDeficiencyFactor( 16 );
	phases[1].SetSquareValuesFactor( 1 );
	phases[1].SetKingSafetyFactor( 0 );
	phases[1].SetTropismFactor( 10 );
	
	//	settings for phase 2 (endgame)
	phases[2].SetNumber( 2 );
	phases[2].SetMobilityFactor( 0 );
	phases[2].SetPawnDeficiencyFactor( 20 );
	phases[2].SetSquareValuesFactor( 1 );
	phases[2].SetKingSafetyFactor( 0 );
	phases[2].SetTropismFactor( 10 );

	// *** OUTPOSTS *** //

	//	we wish to be able to give bonuses to pieces that are 'posted'-
	//	that is, which are centrally located and cannot be reached by
	//	enemy pawns (because adjacent pawns are gone, or have advanced
	//	beyond the piece).  we also wish to be able to give different
	//	bonuses for different types of minor pieces (a posted Knight is
	//	worth more than a posted Bishop.)

	//	we will use the user variable 'gameInt1' in the PieceType class
	//	to store the 'outpost_10x10 factor' for a given piece.  this value will
	//	be multiplied by the square-values stored in the 'outpost_10x10' array.
	DecimalKnight::decimalKnight.gameInt1 = 10;
	DecimalBishop::decimalBishop.gameInt1 = 5;


	// *** INITIALIZATION *** //

	//	initialize hashes
	enPassantHashMap = new HashMap( board.GetNumberOfSquares() );
	castlingHash0k = rand_32();
	castlingHash0q = rand_32();
	castlingHash1k = rand_32();
	castlingHash1q = rand_32();

	//	basic initialization
	board.Initialize( this, BITBOARD_128 );
}

void GrandChessGame::AddPlayerPieceTypes( char *gameName )
{
	//	add the piece types that each player may have
	board.AddPlayerPieceTypeBothPlayers( DecimalRook::decimalRook );
	board.AddPlayerPieceTypeBothPlayers( DecimalBishop::decimalBishop );
	board.AddPlayerPieceTypeBothPlayers( DecimalKnight::decimalKnight );
	board.AddPlayerPieceTypeBothPlayers( DecimalQueen::decimalQueen );
	board.AddPlayerPieceTypeBothPlayers( DecimalKing::decimalKing );
	board.AddPlayerPieceTypeBothPlayers( GrandChessPawn::grandChessPawn );
	board.AddPlayerPieceTypeBothPlayers( DecimalChancellor::decimalChancellor );
	board.AddPlayerPieceTypeBothPlayers( DecimalArchbishop::decimalArchbishop );
}

void GrandChessGame::ChangeRulesByVariant
	( char *gameName,
	  char *&array,
	  char *&book )
{
	//	clear out all piece references
	for( int player = 0; player < 2; player++ )
	{
		queens[player] = NULL;
		marshalls[player] = NULL;
		cardinals[player] = NULL;
		for( int x = 0; x < 4; x++ )
		{
			minorPieces[player][x] = NULL;
			if( x < 3 )
			{
				majorPieces[player][x] = NULL;
				if( x < 2 )
				{
					rooks[player][x] = NULL;
					knights[player][x] = NULL;
					bishops[player][x] = NULL;
				}
			}
		}
	}
	for( int i = 0; i < 10; i++ )
	{
		pawns[0][i] = NULL;
		pawns[1][i] = NULL;
	}

	//	place pieces
	array = "r8r/1nbqkmcbn1/pppppppppp/10/10/10/10/PPPPPPPPPP/1NBQKMCBN1/R8R";
	//	set name of opening book
	book = "openings\\GrandChess.txt";
}

Piece *GrandChessGame::AddPiece
	( PieceType &pieceType,
	  int nPlayer,
	  int nRank,
	  int nFile )
{
	Piece *newPiece = Game::AddPiece( pieceType, nPlayer, nRank, nFile );

	if( pieceType == DecimalQueen::decimalQueen )
	{
		queens[nPlayer] = newPiece;
		AddPieceToSet( newPiece, majorPieces, 3 );
	}
	else if( pieceType == DecimalRook::decimalRook )
	{
		AddPieceToSet( newPiece, rooks, 2 );
	}
	else if( pieceType == DecimalChancellor::decimalChancellor )
	{
		marshalls[nPlayer] = newPiece;
		AddPieceToSet( newPiece, majorPieces, 3 );
	}
	else if( pieceType == DecimalArchbishop::decimalArchbishop )
	{
		cardinals[nPlayer] = newPiece;
		AddPieceToSet( newPiece, majorPieces, 3 );
	}
	else if( pieceType == DecimalBishop::decimalBishop )
	{
		AddPieceToSet( newPiece, bishops, 2 );
		AddPieceToSet( newPiece, minorPieces, 4 );
	}
	else if( pieceType == DecimalKnight::decimalKnight )
	{
		AddPieceToSet( newPiece, knights, 2 );
		AddPieceToSet( newPiece, minorPieces, 4 );
	}
	else if( pieceType.IsPawn() )
	{
		AddPieceToSet( newPiece, pawns, 10 );
		if( newPiece->GetPlayerNumber() == 0 )
			newPiece->GetPromotionZone().AddToZone( 70, 99 );
		else
			newPiece->GetPromotionZone().AddToZone( 0, 29 );
	}
	return newPiece;
}

void GrandChessGame::DeletePiece
	( Piece *piece )
{
	if( queens[0] == piece )
		queens[0] = NULL;
	if( queens[1] == piece )
		queens[1] = NULL;
	if( marshalls[0] == piece )
		marshalls[0] = NULL;
	if( marshalls[1] == piece )
		marshalls[1] = NULL;
	if( cardinals[0] == piece )
		cardinals[0] = NULL;
	if( cardinals[1] == piece )
		cardinals[1] = NULL;
	RemovePieceFromSet( piece, rooks, 2 );
	RemovePieceFromSet( piece, knights, 2 );
	RemovePieceFromSet( piece, bishops, 2 );
	RemovePieceFromSet( piece, minorPieces, 4 );
	RemovePieceFromSet( piece, majorPieces, 3 );
	RemovePieceFromSet( piece, pawns, 10 );
}

void GrandChessGame::AddSpecialMoves
	( int currentPlayer, 
	  MovementList &stack, 
	  GameRec &gameRecord, 
	  bool quiescentSearch )
{
	//	check for en passant captures
	Game::AddEnPassantMoves( stack, gameRecord, quiescentSearch );
}

bool GrandChessGame::MoveBeingMade
	( MoveInfo &moveInfo,
	  GameRec &gameRecord )
{
	// *** UPDATE gameInt1 *** //

	//	gameInt1 is a 'user' field in the game records that a game can use
	//	for any purpose.  we will use it to track the forward progress of 
	//	the game, so we can determine which Phase we are in ('opening', 
	//	'midgame', or 'endgame' in our case)

	if( gameRecord.gameInt1 < 16 )
	{
		//	we are in the opening until gameInt1 reaches 16;
		//	in the opening, we increment gameInt1 for the following moves:
		//		- any capture
		//		- the first move of any piece
		//		- any pawn push to rank 6+ (code value of 5+)
		if( gameRecord.materialCaptured > 0 ||
			(gameRecord.pieceMoved->GetFlags() & (FLAGS_HAS_MOVED | FLAGS_HAS_MOVED_TWICE)) == FLAGS_HAS_MOVED ||
			(gameRecord.pieceMoved->IsPawn() && 
				board.ranks[gameRecord.pieceMoved->GetPlayerNumber()][gameRecord.pieceMoved->GetSquareNumber()] >= 4) )
			gameRecord.gameInt1++;
	}

	return true;
}

Phase &GrandChessGame::AdjustEvaluation
	( int &eval,
	  PawnHash *pPawnHash )
{
	Phase *currentPhase = phases;

	if( board.GetCurrentGameRecord().gameInt1 < 16 )
	{
		// *** The OPENING *** //

		//	in the opening we consider the following things:
		//		- give a penalty for moving major pieces in the opening
		//		- give a penalty for blocking center pawns in the opening
		//		- give a penalty for moving the king to the back rank between 
		//			the rooks
		//		- give a bonus for moving the king to the back rank if not 
		//			between the rooks (the Grand Chess equivilant of castling)

		//	use this "factor" to scale down the adjustments to be applied here, 
		//	so that the adjustments become smaller and smaller as we leave the 
		//	opening and enter the mid-game
		int factor = MAX(board.GetCurrentGameRecord().gameInt1 - 6, 0);

		int factor6 = 640 - (factor << 6);	//   \ 
		int factor5 = 320 - (factor << 5);  //    \_  different adjustments
		int factor4 = 160 - (factor << 4);  //    /   from large to small
		int factor3 = 80 - (factor << 3);   //   / 

		//	give a penalty for moving major pieces (marshall, archbishop, and queen)
		for( int i = 0; i < 3; i++ )
		{
			if( majorPieces[0][i] != NULL && (majorPieces[0][i]->GetFlags() & FLAGS_HAS_MOVED) )
				eval -= factor6;
			if( majorPieces[1][i] != NULL && (majorPieces[1][i]->GetFlags() & FLAGS_HAS_MOVED) )
				eval += factor6;
		}

		//	castling bonuses/penalties
		if( board.ranks[0][board.GetKing( 0 )->GetSquareNumber()] == 0 )
		{
			if( board.bb128_rays[rooks[0][0]->GetSquareNumber() * 100 + 
				rooks[0][1]->GetSquareNumber()] & BitBoard128::GetPositionBitBoard( 
				board.GetKing( 0 )->GetSquareNumber() ) )
				//	king is between the rooks on the back rank; penalty!
				eval -= factor6;
			else
				//	castling bonus!
				eval += factor5;
		}
		if( board.ranks[1][board.GetKing( 1 )->GetSquareNumber()] == 0 )
		{
			if( board.bb128_rays[rooks[1][0]->GetSquareNumber() * 100 + 
				rooks[1][1]->GetSquareNumber()] & BitBoard128::GetPositionBitBoard( 
				board.GetKing( 1 )->GetSquareNumber() ) )
				//	king is between the rooks on the back rank; penalty!
				eval += factor6;
			else
				//	castling bonus!
				eval -= factor5;
		}

		if( testingPosition )
			eval += PenalizePiecesForMovingMultipleTimes();

		//	don't block the king or queen's pawns in the opening!!!
		if( board.GetSquareContents( 34 ) != NULL &&
			board.GetSquareContents( 24 ) != NULL &&
			!board.GetSquareContents( 24 )->HasMoved() )
			eval -= factor5;
		if( board.GetSquareContents( 35 ) != NULL &&
			board.GetSquareContents( 25 ) != NULL &&
			!board.GetSquareContents( 25 )->HasMoved() )
			eval -= factor5;
		if( board.GetSquareContents( 64 ) != NULL &&
			board.GetSquareContents( 74 ) != NULL &&
			!board.GetSquareContents( 74 )->HasMoved() )
			eval += factor5;
		if( board.GetSquareContents( 65 ) != NULL &&
			board.GetSquareContents( 75 ) != NULL &&
			!board.GetSquareContents( 75 )->HasMoved() )
			eval += factor5;
	}
	else if( board.GetNumberOfCapturedPieces() >= 30 || 
		board.GetMaterial( 0 ) - board.GetPawnMaterial( 0 ) + 
		board.GetMaterial( 1 ) - board.GetPawnMaterial( 1 ) < 20000 )
		//	we are in the endgame
		currentPhase = phases + 1;
	else
		//	we are in the mid-game
		currentPhase = phases + 1;


	// *** OUTPOSTS *** //

	//	outposts are minor pieces in central squares that cannot be driven off by enemy pawns.
	//	the basic output bonus is the value of the piece's square in the 'outpost_10x10' array, times
	//	the value of the piece type's 'outpost_10x10 factor' (stored in 'gameInt1'.)

	//	a posted piece that is protected by one of its own pawns is extra good, so
	//	it gets 150% of the above bonus.  a posted piece protected by two of its
	//	own pawns is really, really good, because if the posted piece is captured,
	//	the player will have a protected, passed pawn.  a posted piece that is thus
	//	protected by 2 pawns gets double the standard output bonus.

	//	loop through all minor pieces
	if( pPawnHash != NULL )
	{
		for( int x = 0; x < 4; x++ )
		{
			if( minorPieces[0][x] != NULL &&
				!minorPieces[0][x]->IsCaptured() &&
				outpost_10x10[minorPieces[0][x]->GetSquareNumber()] > 0 &&
				minorPieces[0][x]->GetType().gameInt1 > 0 &&
				((int) (pPawnHash->nPawnsPerFile[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]-1]) == 0 ||
				(int) (pPawnHash->backPawnRank[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]-1]) <= board.ranks[0][minorPieces[0][x]->GetSquareNumber()]) &&
				((int) (pPawnHash->nPawnsPerFile[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]+1]) == 0 ||
				(int) (pPawnHash->backPawnRank[1][board.files[0][minorPieces[0][x]->GetSquareNumber()]+1]) <= board.ranks[0][minorPieces[0][x]->GetSquareNumber()]) )
			{
				//	basic bonus for player 0's posted piece
				eval += outpost_10x10[minorPieces[0][x]->GetSquareNumber()] *
						minorPieces[0][x]->GetType().gameInt1;
				int pawnSquare1 = board.GetMovementMatrix( DIRECTION_SE )[minorPieces[0][x]->GetSquareNumber()];
				int pawnSquare2 = board.GetMovementMatrix( DIRECTION_SW )[minorPieces[0][x]->GetSquareNumber()];
				if( board.GetSquareContents( pawnSquare1 ) != NULL &&
					board.GetSquareContents( pawnSquare1 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval += (outpost_10x10[minorPieces[0][x]->GetSquareNumber()] *
							minorPieces[0][x]->GetType().gameInt1) >> 1;
				if( board.GetSquareContents( pawnSquare2 ) != NULL &&
					board.GetSquareContents( pawnSquare2 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval += (outpost_10x10[minorPieces[0][x]->GetSquareNumber()] *
							minorPieces[0][x]->GetType().gameInt1) >> 1;
			}
			if( minorPieces[1][x] != NULL &&
				!minorPieces[1][x]->IsCaptured() &&
				outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] > 0 &&
				minorPieces[1][x]->GetType().gameInt1 > 0 &&
				((int) (pPawnHash->nPawnsPerFile[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]-1]) == 0 ||
				(int) (pPawnHash->backPawnRank[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]-1]) >= board.ranks[0][minorPieces[1][x]->GetSquareNumber()]) &&
				((int) (pPawnHash->nPawnsPerFile[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]+1]) == 0 ||
				(int) (pPawnHash->backPawnRank[0][board.files[0][minorPieces[1][x]->GetSquareNumber()]+1]) >= board.ranks[0][minorPieces[1][x]->GetSquareNumber()]) )
			{
				//	basic bonus for player 1's posted piece
				eval -= outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] *
						minorPieces[1][x]->GetType().gameInt1;
				int pawnSquare1 = board.GetMovementMatrix( DIRECTION_NE )[minorPieces[1][x]->GetSquareNumber()];
				int pawnSquare2 = board.GetMovementMatrix( DIRECTION_NW )[minorPieces[1][x]->GetSquareNumber()];
				if( board.GetSquareContents( pawnSquare1 ) != NULL &&
					board.GetSquareContents( pawnSquare1 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval -= (outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] *
							minorPieces[1][x]->GetType().gameInt1) >> 1;
				if( board.GetSquareContents( pawnSquare2 ) != NULL &&
					board.GetSquareContents( pawnSquare2 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval -= (outpost_10x10[board.flipSquare[1][minorPieces[1][x]->GetSquareNumber()]] *
							minorPieces[1][x]->GetType().gameInt1) >> 1;
			}
		}
	}

	// *** ROOK ON OPEN and SEMI-OPEN FILE BONUS *** //
	if( pPawnHash != NULL )
	{
		if( rooks[0][0] != NULL &&
			!rooks[0][0]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[0][board.files[0][rooks[0][0]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval += 50;
			if( pPawnHash->nPawnsPerFile[1][board.files[0][rooks[0][0]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval += 20;
		}
		if( rooks[0][1] != NULL &&
			!rooks[0][1]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[0][board.files[0][rooks[0][1]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval += 50;
			if( pPawnHash->nPawnsPerFile[1][board.files[0][rooks[0][1]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval += 20;
		}

		if( rooks[1][0] != NULL &&
			!rooks[1][0]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[1][board.files[0][rooks[1][0]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval -= 50;
			if( pPawnHash->nPawnsPerFile[0][board.files[0][rooks[1][0]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval -= 20;
		}
		if( rooks[1][1] != NULL &&
			!rooks[1][1]->IsCaptured() &&
			pPawnHash->nPawnsPerFile[1][board.files[0][rooks[1][1]->GetSquareNumber()]] == 0 )
		{
			//	file at least semi-open
			eval -= 50;
			if( pPawnHash->nPawnsPerFile[0][board.files[0][rooks[1][1]->GetSquareNumber()]] == 0 )
				//	fully open file
				eval -= 20;
		}
	}

	//	TWO-BISHOPS BONUS
	if( bishops[0][0] != NULL && bishops[0][1] != NULL &&
		!bishops[0][0]->IsCaptured() && !bishops[0][1]->IsCaptured() )
		eval += 150;
	if( bishops[1][0] != NULL && bishops[1][1] != NULL && 
		!bishops[1][0]->IsCaptured() && !bishops[1][1]->IsCaptured() )
		eval -= 150;

	return *currentPhase;
}

int GrandChessGame::EnumeratePromotions
	( Piece *piece, 
	  int fromSquare,
	  int toSquare,
	  Piece **pieces,
	  bool quiescentSearch )
{
	int count = 0;

	//	in Grand Chess, we may only promote to the type of a friendly 
	//	piece that has been captured (for which it is exchanged.)
	
	if( toSquare >= 10 && toSquare < 90 )
		//	promotion is optional, so push "exchange" to same piece
		pieces[count++] = piece;

	bool rookFound = false;
	bool knightFound = false;
	bool bishopFound = false;
	bool queenFound = false;
	bool cardinalFound = false;
	bool marshallFound = false;

	for( int x = 0; x < board.GetNumberOfPieces( piece->GetPlayerNumber() ); x++ )
	{
		Piece *currentPiece = board.GetPiece( piece->GetPlayerNumber(), x );
		if( currentPiece->IsCaptured() )
		{
			if( currentPiece->GetType() == DecimalRook::decimalRook && !rookFound )
			{
				rookFound = true;
				pieces[count++] = currentPiece;
			}
			else if( currentPiece->GetType() == DecimalKnight::decimalKnight && !knightFound )
			{
				knightFound = true;
				pieces[count++] = currentPiece;
			}
			else if( currentPiece->GetType() == DecimalBishop::decimalBishop && !bishopFound )
			{
				bishopFound = true;
				pieces[count++] = currentPiece;
			}
			else if( currentPiece->GetType() == DecimalQueen::decimalQueen && !queenFound )
			{
				queenFound = true;
				pieces[count++] = currentPiece;
			}
			else if( currentPiece->GetType() == DecimalArchbishop::decimalArchbishop && !cardinalFound )
			{
				cardinalFound = true;
				pieces[count++] = currentPiece;
			}
			else if( currentPiece->GetType() == DecimalChancellor::decimalChancellor && !marshallFound )
			{
				marshallFound = true;
				pieces[count++] = currentPiece;
			}
		}
	}

	return count;
}

word32 GrandChessGame::AdjustPrimaryHash
	( word32 primaryHash )
{
	return primaryHash ^ GetEnPassantHash( board.GetCurrentGameRecord(), enPassantHashMap );
}

void GrandChessGame::DefaultSettings()
{
	squareColor1 = RGB(255, 255, 204);
	squareColor2 = RGB(93, 126, 126);
	LoadSquareTextureBitmaps( 0, "Orange Marble" );
	LoadSquareTextureBitmaps( 1, "Wine Marble" );
	pieceColor1 = RGB(255, 255, 255);
	pieceColor2 = RGB(255, 255, 182);
	borderColor = RGB(90, 25, 27);
	boardDisplayType = BOARD_IS_CHECKERED;
	selectedPieceSet = PIECE_SET_STANDARD;
}

void GrandChessGame::SetPersonality
	( int personality )
{
	//	call base class implementation of SetPersonality
	Game::SetPersonality( personality );

	if( personality >= PERSONALITY_C )
	{
		piecesHaveChangingValues = false;
		phases[0].SetPawnDeficiencyFactor( 16 );
		phases[1].SetPawnDeficiencyFactor( 20 );
		phases[2].SetPawnDeficiencyFactor( 24 );
	}
	else
	{
		piecesHaveChangingValues = true;
		phases[0].SetPawnDeficiencyFactor( 12 );
		phases[1].SetPawnDeficiencyFactor( 16 );
		phases[2].SetPawnDeficiencyFactor( 20 );
	}
}
