
/***************************************************************************

                                 ChessV

                   COPYRIGHT (C) 2005 BY GREGORY STRONG

This file is part of ChessV.  ChessV is free software; you can redistribute
it and/or modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of the License, 
or (at your option) any later version.

ChessV is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
more details; the file 'COPYING' contains the License text, but if for
some reason you need a copy, please write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/


#ifndef FILE__CAPABLANCA_GAME_H
#define FILE__CAPABLANCA_GAME_H


#include "../../Board.h"
#include "../../Player.h"
#include "../../PieceType.h"
#include "../../Game.h"
#include "Capablanca_Types.h"


//	****************************
//	**                        **
//	**     CapablancaGame     **
//	**                        **
//	****************************
//
//	this class supports several very similar historical variants of Chess 
//	played on a board with 10 files instead of 8.  the rules of the game
//	are almost identical in almost all cases.
//
//	the CapablancaGame class overrides the Game class to provide all the extra 
//	capabilities needed to play these games.  the Universal Chess Engine 
//	provided supports almost all the rules of Chess natively: move, capture, 
//	promotion, checkmate, stalemate, repetition-draw, 50-move draw, ... 
//	but the ChessGame class overrides virtual functions to provide support 
//	for the following "special rules":  Castling and En Passant 
//	

class CapablancaGame:
	public Game
{
  protected:
	//	data members
	PieceType96 *rookPieceTypes[2];
	PieceType96 *minorPieceTypes[2][4];
	PieceType96 *colorboundPieceTypes[2][2];
	PieceType96 *majorPieceTypes[2][3];
	PieceType96 *pawnPieceTypes[2];
	
	HashMap *enPassantHashMap;
	//	Zobrist keys which are x-or'ed into the board's primary hash
	//	for the current "en passant" square, if any

	word32 castlingHash0k;	//	\   Zobrist keys for castling privledge.
	word32 castlingHash0q;	//	 \_ these values are x-or'ed in if the
	word32 castlingHash1k;	//	 /  player has not lost the ability to
	word32 castlingHash1q;	//	/   castle on the given side.

	//	variant information
	bool enPassant;			//	does this game have the en-passant rule?
	int castlingType;		//	castling type (none, standard, long, flexible, close-rook)
	int pawnPromotionType;	//	promotion type (standard, major)
	bool bareKingRule;		//	Shatranj 'bare king' rule in effect?
	bool greatShatranj;		//	true only if playing Great Shatranj
	bool extendedPieces;

	//	useful bitboards
	BitBoard96 secondRank;
	BitBoard96 seventhRank;
	BitBoard96 whiteSquares;
	BitBoard96 blackSquares;
	BitBoard96 minorStartingSquares[2];


  public:
	// *** INITIALIZATION *** //
	CapablancaGame( Board &board, Player &whitePlayer, Player &blackPlayer );

	virtual void AddPlayerPieceTypes( char *gameName );

	virtual void ChangeRulesByVariant( char *gameName, char *&array, char *&book );

	virtual void ChangeRulesByVariableDefinition( char *gameName );

	virtual void InitializationComplete( char *gameName );


	// ******************************* //
	// ***                         *** //
	// ***     VIRTUAL METHODS     *** //
	// ***                         *** //
	// ******************************* //

	virtual void SetPersonality
		( int personality );

	virtual bool RemovePieceType
		( PieceType *pieceType );

	virtual bool AddPieceType
		( PieceType *pieceType,
		  int nPlayer );

	virtual bool MoveBeingMade
		( MoveInfo &moveInfo,
		  GameRec &gameRecord );

	virtual void AddSpecialMoves
		( int currentPlayer, 
		  MovementList &stack,
		  GameRec &gameRecord, 
		  bool quiescentSearch );

	virtual bool TestForWinLossDraw
		( int &eval );

	virtual word32 AdjustPrimaryHash
		( word32 primaryHash );

	virtual Phase &AdjustEvaluation
		( int &eval,
		  PawnHash *pPawnHash );

	virtual void GenerateMoves
		( int currentPlayer, 
		  MovementList &list );

	virtual void GenerateCaptures
		( int currentPlayer, 
		  MovementList &list );

	virtual bool IsSquareAttacked
		( int squareNumber, 
		  int playerNumber );

	virtual Piece *AddPiece
		( PieceType &pieceType,
		  int nPlayer,
		  int nRank,
		  int nFile );

	virtual int EnumeratePromotions
		( Piece *piece,
		  int fromSquare,
		  int toSquare,
		  PieceType **promotions,
		  bool quiescentSearch );

	virtual int EnumeratePromotions
		( Piece *piece, 
		  int fromSquare,
		  int toSquare,
		  Piece **pieces,
		  bool quiescentSearch );

	virtual void DefaultSettings();
};


#endif
