
/***************************************************************************

                                 ChessV

                   COPYRIGHT (C) 2006 BY GREGORY STRONG

This file is part of ChessV.  ChessV is free software; you can redistribute
it and/or modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of the License, 
or (at your option) any later version.

ChessV is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
more details; the file 'COPYING' contains the License text, but if for
some reason you need a copy, please write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/


#include "StdAfx.h"
#include "Game_12x8.h"
#include "../../ChessV.h"
#include "../../PieceType.h"
#include "../../Piece.h"
#include "../../Direction.h"
#include "../../Rand.h"
#include "../../GameParameters.h"


#include "12x8_Data.h"
//	this include file contains data matricies for square bonuses,
//	outpost bonuses, etc.  it also contains some macros used here.


// *** DEBUG MEMORY ALLOCATION *** //
#ifdef _DEBUG
#define new SAFE_NEW
#endif


//	instances of the PieceTypes
King_12x8 King_12x8::king_12x8;
Pawn_12x8 Pawn_12x8::pawn_12x8;
Knight_12x8 Knight_12x8::knight_12x8;
Bishop_12x8 Bishop_12x8::bishop_12x8;
Rook_12x8 Rook_12x8::rook_12x8;
Queen_12x8 Queen_12x8::queen_12x8;
Archbishop_12x8 Archbishop_12x8::archbishop_12x8;
Chancellor_12x8 Chancellor_12x8::chancellor_12x8;
Amazon_12x8 Amazon_12x8::amazon_12x8;
LamePawn_12x8 LamePawn_12x8::lamePawn_12x8;
General_12x8 General_12x8::general_12x8;
Lion_12x8 Lion_12x8::lion_12x8;
Wazir_12x8 Wazir_12x8::wazir_12x8;
Elephant_12x8 Elephant_12x8::elephant_12x8;
Ferz_12x8 Ferz_12x8::ferz_12x8;
Camel_12x8 Camel_12x8::camel_12x8;
Sage_12x8 Sage_12x8::sage_12x8;
Tower_12x8 Tower_12x8::tower_12x8;
DragonKing_12x8 DragonKing_12x8::dragonKing_12x8;
DragonHorse_12x8 DragonHorse_12x8::dragonHorse_12x8;
Squirrel_12x8 Squirrel_12x8::squirrel_12x8;
KnightGeneral_12x8 KnightGeneral_12x8::knightGeneral_12x8;
Cleric_12x8 Cleric_12x8::cleric_12x8;


Game *CreateGame_12x8( Board &brd, Player &plr0, Player &plr1 )
{
	return new Game_12x8( brd, plr0, plr1 );
}

Game *CreateCourierChessGame( Board &brd, Player &plr0, Player &plr1 )
{
	return new Game_12x8( brd, plr0, plr1 );
}

Game_12x8::Game_12x8( Board &board, Player &whitePlayer, Player &blackPlayer ):
	Game(board, whitePlayer, blackPlayer),
	enPassant(true)
{
	// ***************************************** //
	// ***                                   *** //
	// ***  set rules and evaluation values  *** //
	// ***                                   *** //
	// ***************************************** //

	//	implement 50-move draw rule (100 half-moves)
	autoDrawPeriod = 100;

	//	turn on use of pawn structure evaluation
	usePawnStructureEvaluation = true;

	//	turn off null-move and futility pruning 
	//	after 34 pieces have been captured
	endgameCaptureThreshold = 34;

	//	razoring and futility pruning margins
	razorMargin = 9500;
	futilityMargin = 2750;
	extendedFutilityMargin = 5800;

	// *** PHASES *** //

	//	we have to set up the phases we wish to have in this game;
	//	we will keep it simple and have 3: opening, midgame, and endgame
	nPhases = 3;

	//	settings for phase 0 (opening)
	phases[0].SetNumber( 0 );
	phases[0].SetMobilityFactor( 1 );
	phases[0].SetPawnDeficiencyFactor( 10 );
	phases[0].SetSquareValuesFactor( 1 );
	phases[0].SetKingSafetyFactor( 4 );
	phases[0].SetTropismFactor( 0 );

	//	settings for phase 1 (midgame)
	phases[1].SetNumber( 1 );
	phases[1].SetMobilityFactor( 1 );
	phases[1].SetPawnDeficiencyFactor( 12 );
	phases[1].SetSquareValuesFactor( 1 );
	phases[1].SetKingSafetyFactor( 8 );
	phases[1].SetTropismFactor( 10 );
	
	//	settings for phase 2 (endgame)
	phases[2].SetNumber( 2 );
	phases[2].SetMobilityFactor( 0 );
	phases[2].SetPawnDeficiencyFactor( 16 );
	phases[2].SetSquareValuesFactor( 1 );
	phases[2].SetKingSafetyFactor( 0 );
	phases[2].SetTropismFactor( 10 );

	// *** OUTPOSTS *** //

	//	we wish to be able to give bonuses to pieces that are 'posted'-
	//	that is, which are centrally located and cannot be reached by
	//	enemy pawns (because adjacent pawns are gone, or have advanced
	//	beyond the piece).  we also wish to be able to give different
	//	bonuses for different types of minor pieces (a posted Knight is
	//	worth more than a posted Bishop.)

	//	we will use the user variable 'gameInt1' in the PieceType class
	//	to store the 'outpost factor' for a given piece.  this value will
	//	be multiplied by the square-values stored in the 'outpost_10x8' array.
	Knight_12x8::knight_12x8.gameInt1 = 10;
	Bishop_12x8::bishop_12x8.gameInt1 = 7;
	Lion_12x8::lion_12x8.gameInt1 = 8;
	General_12x8::general_12x8.gameInt1 = 8;

	customMoveGeneration = false;
	customCaptureGeneration = false;

	// *** INITIALIZATION *** //
	for( int x = 0; x < 12; x++ )
	{
		secondRank.SetBit( x + 12 );
		seventhRank.SetBit( x + 72 );
	}

	//	initialize hashes
	enPassantHashMap = new HashMap( board.GetNumberOfSquares() );
	castlingHash0k = rand_32();
	castlingHash0q = rand_32();
	castlingHash1k = rand_32();
	castlingHash1q = rand_32();

	//	basic initialization
	board.Initialize( this, BITBOARD_96 );
}

void Game_12x8::AddPlayerPieceTypes( char *gameName )
{
	//	add the piece types that each player may have
	board.AddPlayerPieceTypeBothPlayers( Rook_12x8::rook_12x8 );
	board.AddPlayerPieceTypeBothPlayers( Bishop_12x8::bishop_12x8 );
	board.AddPlayerPieceTypeBothPlayers( Knight_12x8::knight_12x8 );
	board.AddPlayerPieceTypeBothPlayers( Queen_12x8::queen_12x8 );
	board.AddPlayerPieceTypeBothPlayers( King_12x8::king_12x8 );
	board.AddPlayerPieceTypeBothPlayers( Pawn_12x8::pawn_12x8 );
	board.AddPlayerPieceTypeBothPlayers( Chancellor_12x8::chancellor_12x8 );
	board.AddPlayerPieceTypeBothPlayers( Archbishop_12x8::archbishop_12x8 );
	//	add piece type references to the following arrays, 
	//	minorPieceTypes, etc., used for giving bonuses and 
	//	penalties in positional evaluation
	rookPieceTypes[0] = rookPieceTypes[1] = &Rook_12x8::rook_12x8;
	colorboundPieceTypes[0][0] = colorboundPieceTypes[1][0] = &Bishop_12x8::bishop_12x8;
	colorboundPieceTypes[0][1] = colorboundPieceTypes[1][1] = &NullPieceType96::nullPieceType96;
	colorboundPieceTypes[0][2] = colorboundPieceTypes[1][2] = &NullPieceType96::nullPieceType96;
	minorPieceTypes[0][0] = minorPieceTypes[1][0] = &Knight_12x8::knight_12x8;
	minorPieceTypes[0][1] = minorPieceTypes[1][1] = &Bishop_12x8::bishop_12x8;
	minorPieceTypes[0][2] = minorPieceTypes[1][2] = &NullPieceType96::nullPieceType96;
	minorPieceTypes[0][3] = minorPieceTypes[1][3] = &NullPieceType96::nullPieceType96;
	minorPieceTypes[0][4] = minorPieceTypes[1][4] = &NullPieceType96::nullPieceType96;
	minorPieceTypes[0][5] = minorPieceTypes[1][5] = &NullPieceType96::nullPieceType96;
	majorPieceTypes[0][0] = majorPieceTypes[1][0] = &Queen_12x8::queen_12x8;
	majorPieceTypes[0][1] = majorPieceTypes[1][1] = &Archbishop_12x8::archbishop_12x8;
	majorPieceTypes[0][2] = majorPieceTypes[1][2] = &Chancellor_12x8::chancellor_12x8;
	majorPieceTypes[0][3] = majorPieceTypes[1][3] = &NullPieceType96::nullPieceType96;
	pawnPieceTypes[0] = pawnPieceTypes[1] = &Pawn_12x8::pawn_12x8;
}

// *** CHANGE RULES FOR PARTICULAR (BUILT-IN) VARIANT *** //
void Game_12x8::ChangeRulesByVariant
	( char *gameName,
	  char *&array,
	  char *&book )
{
	//	assume for now standard castling and other rules
	castlingType = CASTLING_TYPE_STANDARD;
	pawnPromotionType = PAWN_PROMOTION_TYPE_STANDARD;
	enPassant = true;

	// *** SOSARIAN CHESS *** //
	if( !strcmp( selectedVariant->name, "Sosarian Chess" ) )
	{
		//	change rules
		castlingType = CASTLING_TYPE_LONG_VERY_CLOSE_ROOK;
		//	add support for the Lion and Cleric
		board.AddPlayerPieceTypeBothPlayers( Lion_12x8::lion_12x8 );
		board.AddPlayerPieceTypeBothPlayers( Cleric_12x8::cleric_12x8 );
		minorPieceTypes[0][2] = minorPieceTypes[1][2] = &Lion_12x8::lion_12x8;
		minorPieceTypes[0][3] = minorPieceTypes[1][3] = &Cleric_12x8::cleric_12x8;
		//	remove Archbishop and Chancellor
		board.RemovePieceType( Archbishop_12x8::archbishop_12x8 );
		board.RemovePieceType( Chancellor_12x8::chancellor_12x8 );
		majorPieceTypes[0][1] = majorPieceTypes[1][1] = &NullPieceType96::nullPieceType96;
		majorPieceTypes[0][2] = majorPieceTypes[1][2] = &NullPieceType96::nullPieceType96;
		//	place pieces
		array = "clrnbqkbnrlc/pppppppppppp/12/12/12/12/PPPPPPPPPPPP/CLRNBQKBNRLC";
		//	set name of opening book
		book = "openings\\SosarianChess.txt";
	}
	// *** COURIER CHESS *** //
	else if( !strcmp( selectedVariant->name, "Courier Chess" ) )
	{
		//	change rules
		castlingType = CASTLING_TYPE_NONE;
		enPassant = false;
		LamePawn_12x8::lamePawn_12x8.SetPromotionType( PromoteToSpecificType );
		LamePawn_12x8::lamePawn_12x8.SetTypeToPromoteTo( Ferz_12x8::ferz_12x8 );
		//	rename pieces
		Bishop_12x8::bishop_12x8.SetFullName( "Courier" );
		Bishop_12x8::bishop_12x8.SetNotation( "C" );
		Elephant_12x8::elephant_12x8.SetFullName( "Bishop" );
		Elephant_12x8::elephant_12x8.SetNotation( "B" );
		Ferz_12x8::ferz_12x8.SetFullName( "Queen" );
		Ferz_12x8::ferz_12x8.SetNotation( "Q" );
		General_12x8::general_12x8.SetFullName( "Man" );
		General_12x8::general_12x8.SetNotation( "M" );
		Wazir_12x8::wazir_12x8.SetFullName( "Schleich" );
		Wazir_12x8::wazir_12x8.SetNotation( "S" );
		//	remove unused piece types
		board.RemovePieceType( Queen_12x8::queen_12x8 );
		board.RemovePieceType( Pawn_12x8::pawn_12x8 );
		//	add new piece types
		board.AddPlayerPieceTypeBothPlayers( LamePawn_12x8::lamePawn_12x8 );
		board.AddPlayerPieceTypeBothPlayers( Ferz_12x8::ferz_12x8 );
		board.AddPlayerPieceTypeBothPlayers( Wazir_12x8::wazir_12x8 );
		board.AddPlayerPieceTypeBothPlayers( General_12x8::general_12x8 );
		board.AddPlayerPieceTypeBothPlayers( Elephant_12x8::elephant_12x8 );
		//	change types in reference arrays
		minorPieceTypes[0][2] = minorPieceTypes[1][2] = &Elephant_12x8::elephant_12x8;
		minorPieceTypes[0][3] = minorPieceTypes[1][3] = &Ferz_12x8::ferz_12x8;
		minorPieceTypes[0][4] = minorPieceTypes[1][4] = &Wazir_12x8::wazir_12x8;
		majorPieceTypes[0][0] = majorPieceTypes[1][0] = &NullPieceType96::nullPieceType96;
		majorPieceTypes[0][1] = majorPieceTypes[1][1] = &NullPieceType96::nullPieceType96;
		majorPieceTypes[0][2] = majorPieceTypes[1][2] = &NullPieceType96::nullPieceType96;
		colorboundPieceTypes[0][1] = colorboundPieceTypes[1][1] = &Elephant_12x8::elephant_12x8;
		colorboundPieceTypes[0][2] = colorboundPieceTypes[1][2] = &Ferz_12x8::ferz_12x8;
		//	place pieces
		array = "rnbcmk1scbnr/1ppppp1pppp1/6q5/p5p4p/P5P4P/6Q5/1PPPPP1PPPP1/RNBCMK1SCBNR";
		//	set name of opening book
		book = "openings\\CourierChess.txt";
	}
	// *** CAGLIOSTRO'S CHESS *** //
	else if( !strcmp( selectedVariant->name, "Cagliostro's Chess" ) )
	{
		//	add new piece types
		board.AddPlayerPieceTypeBothPlayers( Amazon_12x8::amazon_12x8 );
		//	change types in reference arrays
		majorPieceTypes[0][3] = majorPieceTypes[1][3] = &Amazon_12x8::amazon_12x8;
		//	change the name of the Amazon
		Amazon_12x8::amazon_12x8.SetFullName( "General" );
		Amazon_12x8::amazon_12x8.SetNotation( "G" );
		//	place pieces
		array = "rnbacqkgabnr/pppppppppppp/12/12/12/12/PPPPPPPPPPPP/RNBACQKGABNR";
		//	set name of opening book
		book = "openings\\Cagliostro'sChess.txt";
	}
	// *** JANUS KAMIL CHESS *** //
	else if( !strcmp( selectedVariant->name, "Janus Kamil Chess" ) )
	{
		//	change rules
		castlingType = CASTLING_TYPE_LONG_CLOSE_ROOK;
		//	remove the chancellor
		board.RemovePieceType( Chancellor_12x8::chancellor_12x8 );
		majorPieceTypes[0][2] = majorPieceTypes[1][2] = &NullPieceType96::nullPieceType96;
		//	change the name of the archbishop
		Archbishop_12x8::archbishop_12x8.SetFullName( "Janus" );
		Archbishop_12x8::archbishop_12x8.SetNotation( "J" );
		//	add the camel
		board.AddPlayerPieceTypeBothPlayers( Camel_12x8::camel_12x8 );
		minorPieceTypes[0][2] = minorPieceTypes[1][2] = &Camel_12x8::camel_12x8;
		colorboundPieceTypes[0][1] = colorboundPieceTypes[1][1] = &Camel_12x8::camel_12x8;
		//	place pieces
		array = "crjnbqkbnjrc/pppppppppppp/12/12/12/12/PPPPPPPPPPPP/CRJNBQKBNJRC";
		//	set name of opening book
		book = "openings\\JanusKamilChess.txt";
	}
	// *** ARCHCOURIER CHESS *** //
	else if( !strcmp( selectedVariant->name, "ArchCourier Chess" ) )
	{
		//	change rules
		castlingType = CASTLING_TYPE_NONE;
		//	change the name of the knight to the horse
		Knight_12x8::knight_12x8.SetFullName( "Horse" );
		Knight_12x8::knight_12x8.SetNotation( "H" );
		//	replace the chancellor with the crowned rook
		board.RemovePieceType( Chancellor_12x8::chancellor_12x8 );
		board.AddPieceType( DragonKing_12x8::dragonKing_12x8 );
		majorPieceTypes[0][2] = majorPieceTypes[1][2] = &DragonKing_12x8::dragonKing_12x8;
		DragonKing_12x8::dragonKing_12x8.SetFullName( "Crowned Rook" );
		DragonKing_12x8::dragonKing_12x8.SetNotation( "C" );
		//	replace the archbishop with the archcourier
		board.RemovePieceType( Archbishop_12x8::archbishop_12x8 );
		board.AddPieceType( DragonHorse_12x8::dragonHorse_12x8 );
		majorPieceTypes[0][1] = majorPieceTypes[1][1] = &DragonHorse_12x8::dragonHorse_12x8;
		DragonHorse_12x8::dragonHorse_12x8.SetFullName( "ArchCourier" );
		DragonHorse_12x8::dragonHorse_12x8.SetNotation( "A" );
		//	add the squirrel
		board.AddPieceType( Squirrel_12x8::squirrel_12x8 );
		//	add the duke
		board.AddPieceType( KnightGeneral_12x8::knightGeneral_12x8 );
		KnightGeneral_12x8::knightGeneral_12x8.SetFullName( "Duke" );
		KnightGeneral_12x8::knightGeneral_12x8.SetNotation( "D" );
		//	add the guard
		board.AddPieceType( General_12x8::general_12x8 );
		General_12x8::general_12x8.SetFullName( "Guard" );
		//	place pieces
		array = "rhabdqksbchr/pppppppppppp/12/12/12/12/PPPPPPPPPPPP/RHABDQKSBCHR";
		//	set name of opening book
		book = "openings\\ArchCourierChess.txt";
	}
}

void Game_12x8::ChangeRulesByVariableDefinition( char *gameName )
{
	// *** CASTLING TYPE *** //
	char *castlingTypeName;
	if( LookupStringParameter( "castling-type", castlingTypeName ) )
	{
		if( stringCompNoCase( castlingTypeName, "none" ) )
			castlingType = CASTLING_TYPE_NONE;
		else if( stringCompNoCase( castlingTypeName, "standard" ) )
			castlingType = CASTLING_TYPE_STANDARD;
		else if( stringCompNoCase( castlingTypeName, "long" ) )
			castlingType = CASTLING_TYPE_STANDARD;
		else if( stringCompNoCase( castlingTypeName, "flexible" ) )
			castlingType = CASTLING_TYPE_FLEXIBLE;
		else if( stringCompNoCase( castlingTypeName, "close-rook" ) )
			castlingType = CASTLING_TYPE_CLOSE_ROOK;
		else if( stringCompNoCase( castlingTypeName, "long close-rook" ) )
			castlingType = CASTLING_TYPE_LONG_CLOSE_ROOK;
		else if( stringCompNoCase( castlingTypeName, "flexible close-rook" ) )
			castlingType = CASTLING_TYPE_FLEXIBLE_CLOSE_ROOK;
		else
			ASSERT(FALSE);
	}
	// *** PAWN PROMOTION TYPE *** //
	char *pawnPromotionTypeName;
	if( LookupStringParameter( "pawn-promotion-type", pawnPromotionTypeName ) )
	{
		if( stringCompNoCase( pawnPromotionTypeName, "standard" ) )
			pawnPromotionType = PAWN_PROMOTION_TYPE_STANDARD;
		else if( stringCompNoCase( pawnPromotionTypeName, "majors" ) )
			pawnPromotionType = PAWN_PROMOTION_TYPE_MAJORS;
	}
	// *** EN PASSANT *** //
	LookupBoolParameter( "en-passant", enPassant );
}

bool Game_12x8::RemovePieceType
	( PieceType *pieceType )
{
	for( int player = 0; player < 2; player++ )
	{
		if( rookPieceTypes[player] == pieceType )
			rookPieceTypes[player] = &NullPieceType96::nullPieceType96;
		if( pawnPieceTypes[player] == pieceType )
			pawnPieceTypes[player] = &NullPieceType96::nullPieceType96;
		for( int x = 0; x < 6; x++ )
		{
			if( minorPieceTypes[player][x] == pieceType )
			{
				for( int y = x; y < 5; y++ )
					minorPieceTypes[player][y] = minorPieceTypes[player][y+1];
				minorPieceTypes[player][5] = &NullPieceType96::nullPieceType96;
			}
			if( x < 4 )
			{
				if( majorPieceTypes[player][x] == pieceType )
				{
					for( int y = x; y < 3; y++ )
						majorPieceTypes[player][y] = majorPieceTypes[player][y+1];
					majorPieceTypes[player][3] = &NullPieceType96::nullPieceType96;
				}
			}
			if( x < 3 )
			{
				if( colorboundPieceTypes[player][x] == pieceType )
				{
					if( x == 0 )
					{
						colorboundPieceTypes[player][0] = colorboundPieceTypes[player][1];
						colorboundPieceTypes[player][1] = colorboundPieceTypes[player][2];
					}
					else if( x == 1 )
						colorboundPieceTypes[player][1] = colorboundPieceTypes[player][2];
					colorboundPieceTypes[player][2] = &NullPieceType96::nullPieceType96;
				}
			}
		}
	}
	return( true );
}

bool Game_12x8::AddPieceType
	( PieceType *pieceType,
	  int nPlayer )
{
	if( pieceType->IsPawn() )
	{
		if( pawnPieceTypes[nPlayer] == &NullPieceType96::nullPieceType96 )
			pawnPieceTypes[nPlayer] = (PieceType96 *) pieceType;
		else
			return false;
	}
	if( pieceType->GetBaseValue() >= 6000 )
	{
		if( majorPieceTypes[nPlayer][0] == &NullPieceType96::nullPieceType96 )
			majorPieceTypes[nPlayer][0] = (PieceType96 *) pieceType;
		else
		{
			if( majorPieceTypes[nPlayer][1] == &NullPieceType96::nullPieceType96 )
				majorPieceTypes[nPlayer][1] = (PieceType96 *) pieceType;
			else
			{
				if( majorPieceTypes[nPlayer][2] == &NullPieceType96::nullPieceType96 )
					majorPieceTypes[nPlayer][2] = (PieceType96 *) pieceType;
				else
				{
					if( majorPieceTypes[nPlayer][3] == &NullPieceType96::nullPieceType96 )
						majorPieceTypes[nPlayer][3] = (PieceType96 *) pieceType;
					else
					{
						if( majorPieceTypes[nPlayer][4] == &NullPieceType96::nullPieceType96 )
							majorPieceTypes[nPlayer][4] = (PieceType96 *) pieceType;
					}
				}
			}
		}
	}
	else if( pieceType->GetBaseValue() <= 4000 )
	{
		if( minorPieceTypes[nPlayer][0] == &NullPieceType96::nullPieceType96 )
			minorPieceTypes[nPlayer][0] = (PieceType96 *) pieceType;
		else
		{
			if( minorPieceTypes[nPlayer][1] == &NullPieceType96::nullPieceType96 )
				minorPieceTypes[nPlayer][1] = (PieceType96 *) pieceType;
			else
			{
				if( minorPieceTypes[nPlayer][2] == &NullPieceType96::nullPieceType96 )
					minorPieceTypes[nPlayer][2] = (PieceType96 *) pieceType;
				else
				{
					if( minorPieceTypes[nPlayer][3] == &NullPieceType96::nullPieceType96 )
						minorPieceTypes[nPlayer][3] = (PieceType96 *) pieceType;
					else
					{
						if( minorPieceTypes[nPlayer][4] == &NullPieceType96::nullPieceType96 )
							minorPieceTypes[nPlayer][4] = (PieceType96 *) pieceType;
						else
						{
							if( minorPieceTypes[nPlayer][5] == &NullPieceType96::nullPieceType96 )
								minorPieceTypes[nPlayer][5] = (PieceType96 *) pieceType;
						}
					}
				}
			}
		}
	}
	return true;
}

Piece *Game_12x8::AddPiece
	( PieceType &pieceType,
	  int nPlayer,
	  int nRank,
	  int nFile )
{
	Piece *newPiece = Game::AddPiece( pieceType, nPlayer, nRank, nFile );

	if( &pieceType == pawnPieceTypes[nPlayer] )
	{
		if( newPiece->GetPlayerNumber() == 0 )
			newPiece->GetPromotionZone().AddToZone( 84, 95 );
		else
			newPiece->GetPromotionZone().AddToZone( 0, 11 );
	}
	return newPiece;
}

void Game_12x8::AddSpecialMoves
	( int currentPlayer, 
	  MovementList &stack,  
	  GameRec &gameRecord, 
	  bool quiescentSearch )
{
	// *** EN PASSANT *** //

	if( enPassant )
		Game::AddEnPassantMoves( stack, gameRecord, quiescentSearch );

	
	//	castling moves are added next; check for an early exit
	if( quiescentSearch || board.FastCheckTest() || castlingType == CASTLING_TYPE_NONE )
		//	return now, so we don't consider castling moves
		//	in quiescent search, or when we are in Check
		return;


	// *** CASTLING *** //

	if( castlingType == CASTLING_TYPE_STANDARD && board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() && king->GetSquareNumber() == 6 )
		{
			//	king side
			if( board.GetSquareContents( 11 ) != NULL &&
				!board.GetSquareContents( 11 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 11 ), 10, 9 );
			}
			//	queen side
			if( board.GetSquareContents( 0 ) != NULL &&
				!board.GetSquareContents( 0 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 0 ), 2, 3 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 5 )
		{
			//	king side
			if( board.GetSquareContents( 11 ) != NULL &&
				!board.GetSquareContents( 11 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 11 ), 9, 8 );
			}
			//	queen side
			if( board.GetSquareContents( 0 ) != NULL &&
				!board.GetSquareContents( 0 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 0 ), 1, 2 );
			}
		}
	}
	else if( castlingType == CASTLING_TYPE_STANDARD && board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() && king->GetSquareNumber() == 90 )
		{
			//	king side
			if( board.GetSquareContents( 95 ) != NULL &&
				!board.GetSquareContents( 95 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 95 ), 94, 93 );
			}
			//	queen side
			if( board.GetSquareContents( 84 ) != NULL &&
				!board.GetSquareContents( 84 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 84 ), 86, 87 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 89 )
		{
			//	king side
			if( board.GetSquareContents( 95 ) != NULL &&
				!board.GetSquareContents( 95 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 95 ), 93, 92 );
			}
			//	queen side
			if( board.GetSquareContents( 84 ) != NULL &&
				!board.GetSquareContents( 84 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 84 ), 85, 86 );
			}
		}
	}
	// *** LONG CASTLING *** //
	else if( castlingType == CASTLING_TYPE_LONG && board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() && king->GetSquareNumber() == 6 )
		{
			//	king side
			if( board.GetSquareContents( 11 ) != NULL &&
				!board.GetSquareContents( 11 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 11 ), 10, 9 );
			}
			//	queen side
			if( board.GetSquareContents( 0 ) != NULL &&
				!board.GetSquareContents( 0 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 0 ), 1, 2 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 5 )
		{
			//	king side
			if( board.GetSquareContents( 11 ) != NULL &&
				!board.GetSquareContents( 11 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 11 ), 10, 9 );
			}
			//	queen side
			if( board.GetSquareContents( 0 ) != NULL &&
				!board.GetSquareContents( 0 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 0 ), 1, 2 );
			}
		}
	}
	else if( castlingType == CASTLING_TYPE_LONG && board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() && king->GetSquareNumber() == 90 )
		{
			//	king side
			if( board.GetSquareContents( 95 ) != NULL &&
				!board.GetSquareContents( 95 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 95 ), 94, 93 );
			}
			//	queen side
			if( board.GetSquareContents( 84 ) != NULL &&
				!board.GetSquareContents( 84 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 84 ), 85, 86 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 89 )
		{
			//	king side
			if( board.GetSquareContents( 95 ) != NULL &&
				!board.GetSquareContents( 95 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 95 ), 94, 93 );
			}
			//	queen side
			if( board.GetSquareContents( 84 ) != NULL &&
				!board.GetSquareContents( 84 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 84 ), 85, 86 );
			}
		}
	}
	// *** CLOSE-ROOK CASTLING *** //
	else if( castlingType == CASTLING_TYPE_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() && king->GetSquareNumber() == 6 )
		{
			//	king side
			if( board.GetSquareContents( 10 ) != NULL &&
				!board.GetSquareContents( 10 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 10 ), 9, 8 );
			}
			//	queen side
			if( board.GetSquareContents( 1 ) != NULL &&
				!board.GetSquareContents( 1 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 1 ), 3, 4 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 5 )
		{
			//	king side
			if( board.GetSquareContents( 10 ) != NULL &&
				!board.GetSquareContents( 10 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 10 ), 8, 7 );
			}
			//	queen side
			if( board.GetSquareContents( 1 ) != NULL &&
				!board.GetSquareContents( 1 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 1 ), 2, 3 );
			}
		}
	}
	else if( castlingType == CASTLING_TYPE_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() && king->GetSquareNumber() == 90 )
		{
			//	king side
			if( board.GetSquareContents( 94 ) != NULL &&
				!board.GetSquareContents( 94 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 94 ), 93, 92 );
			}
			//	queen side
			if( board.GetSquareContents( 85 ) != NULL &&
				!board.GetSquareContents( 85 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 85 ), 87, 88 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 89 )
		{
			//	king side
			if( board.GetSquareContents( 94 ) != NULL &&
				!board.GetSquareContents( 94 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 94 ), 92, 91 );
			}
			//	queen side
			if( board.GetSquareContents( 85 ) != NULL &&
				!board.GetSquareContents( 85 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 85 ), 86, 87 );
			}
		}
	}
	// *** VERY-CLOSE-ROOK CASTLING *** //
	else if( castlingType == CASTLING_TYPE_VERY_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() && king->GetSquareNumber() == 6 )
		{
			//	king side
			if( board.GetSquareContents( 9 ) != NULL &&
				!board.GetSquareContents( 9 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 9 ), 8, 7 );
			}
			//	queen side
			if( board.GetSquareContents( 2 ) != NULL &&
				!board.GetSquareContents( 2 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 2 ), 4, 5 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 5 )
		{
			//	king side
			if( board.GetSquareContents( 9 ) != NULL &&
				!board.GetSquareContents( 9 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 9 ), 7, 6 );
			}
			//	queen side
			if( board.GetSquareContents( 2 ) != NULL &&
				!board.GetSquareContents( 2 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 2 ), 3, 4 );
			}
		}
	}
	else if( castlingType == CASTLING_TYPE_VERY_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() && king->GetSquareNumber() == 90 )
		{
			//	king side
			if( board.GetSquareContents( 93 ) != NULL &&
				!board.GetSquareContents( 93 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 93 ), 92, 91 );
			}
			//	queen side
			if( board.GetSquareContents( 86 ) != NULL &&
				!board.GetSquareContents( 86 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 86 ), 88, 89 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 89 )
		{
			//	king side
			if( board.GetSquareContents( 93 ) != NULL &&
				!board.GetSquareContents( 93 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 94 ), 91, 90 );
			}
			//	queen side
			if( board.GetSquareContents( 86 ) != NULL &&
				!board.GetSquareContents( 86 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 86 ), 87, 88 );
			}
		}
	}
	// *** LONG-VERY-CLOSE-ROOK CASTLING *** //
	else if( castlingType == CASTLING_TYPE_LONG_VERY_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() && king->GetSquareNumber() == 6 )
		{
			//	king side
			if( board.GetSquareContents( 9 ) != NULL &&
				!board.GetSquareContents( 9 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 9 ), 8, 7 );
			}
			//	queen side
			if( board.GetSquareContents( 2 ) != NULL &&
				!board.GetSquareContents( 2 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 2 ), 3, 4 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 5 )
		{
			//	king side
			if( board.GetSquareContents( 9 ) != NULL &&
				!board.GetSquareContents( 9 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 9 ), 8, 7 );
			}
			//	queen side
			if( board.GetSquareContents( 2 ) != NULL &&
				!board.GetSquareContents( 2 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 2 ), 3, 4 );
			}
		}
	}
	else if( castlingType == CASTLING_TYPE_LONG_VERY_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() && king->GetSquareNumber() == 90 )
		{
			//	king side
			if( board.GetSquareContents( 93 ) != NULL &&
				!board.GetSquareContents( 93 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 93 ), 92, 91 );
			}
			//	queen side
			if( board.GetSquareContents( 86 ) != NULL &&
				!board.GetSquareContents( 86 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 86 ), 87, 88 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 89 )
		{
			//	king side
			if( board.GetSquareContents( 93 ) != NULL &&
				!board.GetSquareContents( 93 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 94 ), 92, 91 );
			}
			//	queen side
			if( board.GetSquareContents( 86 ) != NULL &&
				!board.GetSquareContents( 86 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 86 ), 87, 88 );
			}
		}
	}
	// *** LONG CLOSE-ROOK CASTLING *** //
	else if( castlingType == CASTLING_TYPE_LONG_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 0 )
	{
		//	white to move
		Piece *king = board.GetKing( 0 );
		if( !king->HasMoved() && king->GetSquareNumber() == 6 )
		{
			//	king side
			if( board.GetSquareContents( 10 ) != NULL &&
				!board.GetSquareContents( 10 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 10 ), 9, 8 );
			}
			//	queen side
			if( board.GetSquareContents( 1 ) != NULL &&
				!board.GetSquareContents( 1 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 1 ), 2, 3 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 5 )
		{
			//	king side
			if( board.GetSquareContents( 10 ) != NULL &&
				!board.GetSquareContents( 10 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 10 ), 9, 8 );
			}
			//	queen side
			if( board.GetSquareContents( 1 ) != NULL &&
				!board.GetSquareContents( 1 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 1 ), 2, 3 );
			}
		}
	}
	else if( castlingType == CASTLING_TYPE_LONG_CLOSE_ROOK && board.GetCurrentPlayerNumber() == 1 )
	{
		//	black to move
		Piece *king = board.GetKing( 1 );
		if( !king->HasMoved() && king->GetSquareNumber() == 90 )
		{
			//	king side
			if( board.GetSquareContents( 94 ) != NULL &&
				!board.GetSquareContents( 94 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 94 ), 93, 92 );
			}
			//	queen side
			if( board.GetSquareContents( 85 ) != NULL &&
				!board.GetSquareContents( 85 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 85 ), 86, 87 );
			}
		}
		else if( !king->HasMoved() && king->GetSquareNumber() == 89 )
		{
			//	king side
			if( board.GetSquareContents( 94 ) != NULL &&
				!board.GetSquareContents( 94 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 94 ), 93, 92 );
			}
			//	queen side
			if( board.GetSquareContents( 85 ) != NULL &&
				!board.GetSquareContents( 85 )->HasMoved() )
			{
				AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 85 ), 86, 87 );
			}
		}
	}
	// *** FLEXIBLE CASTLING *** //
	else if( castlingType == CASTLING_TYPE_FLEXIBLE )
	{
		if( board.GetCurrentPlayerNumber() == 0 )
		{
			//	white to move
			Piece *king = board.GetKing( 0 );
			if( !king->HasMoved() && king->GetSquareNumber() > 0 && king->GetSquareNumber() < 12 )
			{
				Piece *leftCastlingPiece = NULL;
				Piece *rightCastlingPiece = NULL;
				//	check right castling-piece
				if( board.GetSquareContents( 11 ) != NULL &&
					!board.GetSquareContents( 11 )->HasMoved() )
					//	right castling-piece is castling-elegible
					rightCastlingPiece = board.GetSquareContents( 11 );
				//	check left castling-piece
				if( board.GetSquareContents( 0 ) != NULL &&
					!board.GetSquareContents( 0 )->HasMoved() )
					//	left castling-piece is castling-elegible
					leftCastlingPiece = board.GetSquareContents( 0 );
				Game::AddFlexibleCastlingMoves( stack, gameRecord, king, leftCastlingPiece, rightCastlingPiece );
			}
		}
		else
		{
			//	black to move
			Piece *king = board.GetKing( 1 );
			if( !king->HasMoved() && king->GetSquareNumber() > 84 && king->GetSquareNumber() < 95 )
			{
				Piece *leftCastlingPiece = NULL;
				Piece *rightCastlingPiece = NULL;
				//	check right castling-piece
				if( board.GetSquareContents( 84 ) != NULL &&
					!board.GetSquareContents( 84 )->HasMoved() )
					//	right castling-piece is castling-elegible
					rightCastlingPiece = board.GetSquareContents( 84 );
				//	check left castling-piece
				if( board.GetSquareContents( 95 ) != NULL &&
					!board.GetSquareContents( 95 )->HasMoved() )
					//	left castling-piece is castling-elegible
					leftCastlingPiece = board.GetSquareContents( 95 );
				Game::AddFlexibleCastlingMoves( stack, gameRecord, king, leftCastlingPiece, rightCastlingPiece );
			}
		}
	}
	// *** FLEXIBLE CLOSE-ROOK CASTLING *** //
	else if( castlingType == CASTLING_TYPE_FLEXIBLE_CLOSE_ROOK )
	{
		if( board.GetCurrentPlayerNumber() == 0 )
		{
			//	white to move
			Piece *king = board.GetKing( 0 );
			if( !king->HasMoved() && king->GetSquareNumber() > 1 && king->GetSquareNumber() < 10 )
			{
				Piece *leftCastlingPiece = NULL;
				Piece *rightCastlingPiece = NULL;
				//	check right castling-piece
				if( board.GetSquareContents( 10 ) != NULL &&
					!board.GetSquareContents( 10 )->HasMoved() )
					//	right castling-piece is castling-elegible
					rightCastlingPiece = board.GetSquareContents( 10 );
				//	check left castling-piece
				if( board.GetSquareContents( 1 ) != NULL &&
					!board.GetSquareContents( 1 )->HasMoved() )
					//	left castling-piece is castling-elegible
					leftCastlingPiece = board.GetSquareContents( 1 );
				Game::AddFlexibleCastlingMoves( stack, gameRecord, king, leftCastlingPiece, rightCastlingPiece );
			}
		}
		else
		{
			//	black to move
			Piece *king = board.GetKing( 1 );
			if( !king->HasMoved() && king->GetSquareNumber() > 85 && king->GetSquareNumber() < 94 )
			{
				Piece *leftCastlingPiece = NULL;
				Piece *rightCastlingPiece = NULL;
				//	check right castling-piece
				if( board.GetSquareContents( 94 ) != NULL &&
					!board.GetSquareContents( 94 )->HasMoved() )
					//	right castling-piece is castling-elegible
					rightCastlingPiece = board.GetSquareContents( 94 );
				//	check left castling-piece
				if( board.GetSquareContents( 85 ) != NULL &&
					!board.GetSquareContents( 85 )->HasMoved() )
					//	left castling-piece is castling-elegible
					leftCastlingPiece = board.GetSquareContents( 85 );
				Game::AddFlexibleCastlingMoves( stack, gameRecord, king, leftCastlingPiece, rightCastlingPiece );
			}
		}
	}
}

bool Game_12x8::MoveBeingMade
	( MoveInfo &moveInfo,
	  GameRec &gameRecord )
{
	// *** UPDATE gameInt1 *** //

	//	gameInt1 is a 'user' field in the game records that a game can use
	//	for any purpose.  we will use it to track the forward progress of 
	//	the game, so we can determine which Phase we are in ('opening', 
	//	'midgame', or 'endgame' in our case)

	if( gameRecord.gameInt1 < 20 )
	{
		//	we are in the opening until gameInt1 reaches 20;
		//	in the opening, we increment gameInt1 for the following moves:
		//		- any capture
		//		- the first move of any piece
		//		- any pawn push to rank 5+ (code value of 4+)
		//		- in the special case of castling, add 3 to gameInt1
		if( moveInfo.type == Castling || 
			moveInfo.type == FlexibleCastling || 
			moveInfo.type == FreeCastling || 
			moveInfo.type == UserCastling )
			gameRecord.gameInt1 += 3;
		else if( gameRecord.materialCaptured > 0 ||
			(gameRecord.pieceMoved->GetFlags() & (FLAGS_HAS_MOVED | FLAGS_HAS_MOVED_TWICE)) == FLAGS_HAS_MOVED ||
			(gameRecord.pieceMoved->IsPawn() && 
				board.ranks[gameRecord.pieceMoved->GetPlayerNumber()][gameRecord.pieceMoved->GetSquareNumber()] >= 4) )
			gameRecord.gameInt1++;
	}

	return true;
}

Phase &Game_12x8::AdjustEvaluation
	( int &eval,
	  PawnHash *pPawnHash )
{
	Phase *currentPhase = phases;

	if( board.GetCurrentGameRecord().gameInt1 < 20 )
	{
		// *** The OPENING *** //

		//	we are in the opening until gameInt1 reaches 20;
		//	in the opening, we increment gameInt1 for the following moves:
		//		- any capture
		//		- the first move of any piece
		//		- any pawn push to rank 5+ (code value of 4+)
		//		- in the special case of castling, add 3 to gameInt1

		//	in the opening we consider the following things:
		//		- give a bonus for castling; give a penalty for losing the ability to castle
		//		- give a bonus for developing minor pieces, penalty for moving major pieces early
		//		- penalize any piece for moving twice, with a bigger penalty for moving three times

		//	use this "factor" to scale down the adjustments to be applied here, 
		//	so that the adjustments become smaller and smaller as we leave the 
		//	opening and enter the mid-game
		int factor = MAX(board.GetCurrentGameRecord().gameInt1 - 10, 0);
		int factor6 = 640 - (factor << 6);	//   \ 
		int factor5 = 320 - (factor << 5);  //    \_  different adjustments
		int factor4 = 160 - (factor << 4);  //    /   from large to small
		int factor3 = 80 - (factor << 3);   //   / 

		// *** CASTLING adjustment *** //
		if( castlingType != CASTLING_TYPE_NONE )
		{
			//	if we are playing Paulowich's variant, we need to 
			//	adjust the castling squares of the rooks
			int adjustment = 0;
			if( castlingType == CASTLING_TYPE_CLOSE_ROOK ||
				castlingType == CASTLING_TYPE_FLEXIBLE_CLOSE_ROOK )
				adjustment = 1;

			// *** CASTLING bonus/penalty *** //
			if( board.GetKing( 0 )->GetFlags() & FLAGS_HAS_CASTLED )
				eval += factor5;
			else
			{
				if( board.GetKing( 0 )->HasMoved() || 
					((board.GetSquareContents( 11 - adjustment ) == NULL ||
					  board.GetSquareContents( 11 - adjustment )->HasMoved()) &&
					 (board.GetSquareContents( 0 + adjustment ) == NULL ||
					  board.GetSquareContents( 0 + adjustment )->HasMoved())) )
					eval -= factor5;
			}
			if( board.GetKing( 1 )->GetFlags() & FLAGS_HAS_CASTLED )
				eval -= factor5;
			else
			{
				if( board.GetKing( 1 )->HasMoved() || 
					((board.GetSquareContents( 95 - adjustment ) == NULL ||
					  board.GetSquareContents( 95 - adjustment )->HasMoved()) &&
					 (board.GetSquareContents( 84 + adjustment ) == NULL ||
					  board.GetSquareContents( 84 + adjustment )->HasMoved())) )
					eval += factor5;
			}
		}

		//	give a penalty for the major pieces moving early
		BitBoard96 majors;
		majors = majorPieceTypes[0][0]->GetPieces( 0 ) | 
			     majorPieceTypes[0][1]->GetPieces( 0 ) | 
			     majorPieceTypes[0][2]->GetPieces( 0 ) | 
		         majorPieceTypes[0][3]->GetPieces( 0 );
		while( majors )
		{
			int square = majors.GetFirstBit();
			majors.ToggleBit( square );
			if( board.GetSquareContents( square )->GetFlags() & FLAGS_HAS_MOVED )
				eval -= factor5;
		}
		majors = majorPieceTypes[1][0]->GetPieces( 1 ) | 
			     majorPieceTypes[1][1]->GetPieces( 1 ) | 
			     majorPieceTypes[1][2]->GetPieces( 1 ) | 
				 majorPieceTypes[1][3]->GetPieces( 1 );
		while( majors )
		{
			int square = majors.GetFirstBit();
			majors.ToggleBit( square );
			if( board.GetSquareContents( square )->GetFlags() & FLAGS_HAS_MOVED )
				eval += factor5;
		}

		//	don't block the king's or queen's pawns in the opening!!!
		if( board.GetSquareContents( 29 ) != NULL &&
			board.GetSquareContents( 17 ) != NULL &&
			!board.GetSquareContents( 17 )->HasMoved() )
			eval -= factor6;
		if( board.GetSquareContents( 30 ) != NULL &&
			board.GetSquareContents( 18 ) != NULL &&
			!board.GetSquareContents( 18 )->HasMoved() )
			eval -= factor6;
		if( board.GetSquareContents( 65 ) != NULL &&
			board.GetSquareContents( 77 ) != NULL &&
			!board.GetSquareContents( 77 )->HasMoved() )
			eval += factor6;
		if( board.GetSquareContents( 66 ) != NULL &&
			board.GetSquareContents( 78 ) != NULL &&
			!board.GetSquareContents( 78 )->HasMoved() )
			eval += factor6;
	}
	else if( board.GetNumberOfCapturedPieces() > 34 ||
		board.GetMaterial( 0 ) - board.GetPawnMaterial( 0 ) + 
		board.GetMaterial( 1 ) - board.GetPawnMaterial( 1 ) < 24000 )
		//	we are in the end-game
		currentPhase = phases + 2;
	else
		currentPhase = phases + 1;


	if( pPawnHash != NULL && currentPhase != phases )
	{
		// *** OUTPOSTS *** //

		//	outposts are minor pieces in central squares that cannot be driven off by enemy pawns.
		//	the basic output bonus is the value of the piece's square in the 'outpost' array, times
		//	the value of the piece type's 'outpost factor' (stored in 'gameInt1'.)

		//	a posted piece that is protected by one of its own pawns is extra good, so
		//	it gets 150% of the above bonus.  a posted piece protected by two of its
		//	own pawns is really, really good, because if the posted piece is captured,
		//	the player will have a protected, passed pawn.  a posted piece that is thus
		//	protected by 2 pawns gets double the standard output bonus.

		//	loop through all minor pieces
		BitBoard96 minors;
		minors = minorPieceTypes[0][0]->GetPieces( 0 ) | 
				 minorPieceTypes[0][1]->GetPieces( 0 ) | 
				 minorPieceTypes[0][2]->GetPieces( 0 ) | 
				 minorPieceTypes[0][3]->GetPieces( 0 ) | 
				 minorPieceTypes[0][4]->GetPieces( 0 ) | 
				 minorPieceTypes[0][5]->GetPieces( 0 );
		while( minors )
		{
			int square = minors.GetFirstBit();
			minors.ToggleBit( square );
			Piece *piece = board.GetSquareContents( square );
			if( outpost_12x8[square] > 0 && piece->GetType().gameInt1 > 0 &&
				((int) (pPawnHash->nPawnsPerFile[1][board.files[0][square]-1]) == 0 ||
				(int) (pPawnHash->backPawnRank[1][board.files[0][square]-1]) <= board.ranks[0][square]) &&
				((int) (pPawnHash->nPawnsPerFile[1][board.files[0][square]+1]) == 0 ||
				(int) (pPawnHash->backPawnRank[1][board.files[0][square]+1]) <= board.ranks[0][square]) )
			{
				//	basic bonus for player 0's posted piece
				eval += outpost_12x8[square] * piece->GetType().gameInt1;
				int pawnSquare1 = board.GetMovementMatrix( DIRECTION_SE )[square];
				int pawnSquare2 = board.GetMovementMatrix( DIRECTION_SW )[square];
				if( board.GetSquareContents( pawnSquare1 ) != NULL &&
					board.GetSquareContents( pawnSquare1 )->GetPlayerNumber() == 0 && 
					board.GetSquareContents( pawnSquare1 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval += (outpost_12x8[square] * piece->GetType().gameInt1) >> 1;
				if( board.GetSquareContents( pawnSquare2 ) != NULL &&
					board.GetSquareContents( pawnSquare2 )->GetPlayerNumber() == 0 && 
					board.GetSquareContents( pawnSquare2 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval += (outpost_12x8[square] * piece->GetType().gameInt1) >> 1;
			}
		}
		minors = minorPieceTypes[1][0]->GetPieces( 1 ) | 
				 minorPieceTypes[1][1]->GetPieces( 1 ) | 
				 minorPieceTypes[1][2]->GetPieces( 1 ) | 
				 minorPieceTypes[1][3]->GetPieces( 1 ) | 
				 minorPieceTypes[1][4]->GetPieces( 1 ) | 
				 minorPieceTypes[1][5]->GetPieces( 1 );
		while( minors )
		{
			int square = minors.GetFirstBit();
			minors.ToggleBit( square );
			Piece *piece = board.GetSquareContents( square );
			if( outpost_12x8[board.flipSquare[1][square]] > 0 && piece->GetType().gameInt1 > 0 &&
				((int) (pPawnHash->nPawnsPerFile[0][board.files[0][square]-1]) == 0 ||
				(int) (pPawnHash->backPawnRank[0][board.files[0][square]-1]) <= board.ranks[0][square]) &&
				((int) (pPawnHash->nPawnsPerFile[0][board.files[0][square]+1]) == 0 ||
				(int) (pPawnHash->backPawnRank[0][board.files[0][square]+1]) <= board.ranks[0][square]) )
			{
				//	basic bonus for player 1's posted piece
				eval -= outpost_12x8[board.flipSquare[1][square]] * piece->GetType().gameInt1;
				int pawnSquare1 = board.GetMovementMatrix( DIRECTION_SE )[square];
				int pawnSquare2 = board.GetMovementMatrix( DIRECTION_SW )[square];
				if( board.GetSquareContents( pawnSquare1 ) != NULL && 
					board.GetSquareContents( pawnSquare1 )->GetPlayerNumber() == 1 && 
					board.GetSquareContents( pawnSquare1 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval -= (outpost_12x8[board.flipSquare[1][square]] * piece->GetType().gameInt1) >> 1;
				if( board.GetSquareContents( pawnSquare2 ) != NULL &&
					board.GetSquareContents( pawnSquare2 )->GetPlayerNumber() == 1 && 
					board.GetSquareContents( pawnSquare2 )->GetType().IsPawn() )
					//	additional 50% bonus for this pawn defender
					eval -= (outpost_12x8[board.flipSquare[1][square]] * piece->GetType().gameInt1) >> 1;
			}
		}

		// *** ROOK ON OPEN and SEMI-OPEN FILE BONUS *** //
		BitBoard96 rooks;
		rooks = rookPieceTypes[0]->GetPieces( 0 );
		while( rooks )
		{
			int square = rooks.GetFirstBit();
			rooks.ToggleBit( square );
			if( pPawnHash->nPawnsPerFile[0][board.files[0][square]] == 0 )
			{
				//	file at least semi-open
				eval += 50;
				if( pPawnHash->nPawnsPerFile[1][board.files[0][square]] == 0 )
					//	fully open file
					eval += 20;
			}
		}
		rooks = rookPieceTypes[1]->GetPieces( 1 );
		while( rooks )
		{
			int square = rooks.GetFirstBit();
			rooks.ToggleBit( square );
			if( pPawnHash->nPawnsPerFile[1][board.files[0][square]] == 0 )
			{
				//	file at least semi-open
				eval -= 50;
				if( pPawnHash->nPawnsPerFile[0][board.files[0][square]] == 0 )
					//	fully open file
					eval -= 20;
			}
		}
	}

	//	TWO-BISHOPS BONUS
	//		TODO: temporarily removed

	return *currentPhase;
}

word32 Game_12x8::AdjustPrimaryHash
	( word32 primaryHash )
{
	// *** adjust for castling privledges *** //

	if( castlingType == CASTLING_TYPE_CLOSE_ROOK ||
		castlingType == CASTLING_TYPE_FLEXIBLE_CLOSE_ROOK )
	{
		if( !board.GetKing( 0 )->HasMoved() )
		{
			Piece *castlingPiece = board.GetSquareContents( 1 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() )
				primaryHash ^= castlingHash0q;
			castlingPiece = board.GetSquareContents( 10 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() && 
				&(castlingPiece->GetType()) == rookPieceTypes[0] )
				primaryHash ^= castlingHash0k;
		}
		else if( !board.GetKing( 1 )->HasMoved() )
		{
			Piece *castlingPiece = board.GetSquareContents( 85 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() )
				primaryHash ^= castlingHash1q;
			castlingPiece = board.GetSquareContents( 94 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() && 
				&(castlingPiece->GetType()) == rookPieceTypes[1] )
				primaryHash ^= castlingHash1k;
		}
	}
	else if( castlingType != CASTLING_TYPE_NONE )
	{
		if( !board.GetKing( 0 )->HasMoved() )
		{
			Piece *castlingPiece = board.GetSquareContents( 0 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() )
				primaryHash ^= castlingHash0q;
			castlingPiece = board.GetSquareContents( 11 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() )
				primaryHash ^= castlingHash0k;
		}
		else if( !board.GetKing( 1 )->HasMoved() )
		{
			Piece *castlingPiece = board.GetSquareContents( 84 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() )
				primaryHash ^= castlingHash1q;
			castlingPiece = board.GetSquareContents( 95 );
			if( castlingPiece != NULL && 
				!castlingPiece->HasMoved() )
				primaryHash ^= castlingHash1k;
		}
	}
	
	// *** adjust for en passant square *** //
	if( enPassant )
		return primaryHash ^ GetEnPassantHash( board.GetCurrentGameRecord(), enPassantHashMap );
	return primaryHash;
}

int Game_12x8::EnumeratePromotions
	( Piece *piece, 
	  int fromSquare,
	  int toSquare,
	  PieceType **promotions,
	  bool quiescentSearch )
{
	// *** MAJORS pawn promotion *** //
	if( pawnPromotionType == PAWN_PROMOTION_TYPE_MAJORS )
	{
		//	with this type of pawn promotion, pawns may only 
		//	promote to major piece types
		int promotionTypeCount = 0;
		if( majorPieceTypes[piece->GetPlayerNumber()][0] != &NullPieceType96::nullPieceType96 )
		{
			promotionTypeCount++;
			promotions[0] = majorPieceTypes[piece->GetPlayerNumber()][0];
			if( majorPieceTypes[piece->GetPlayerNumber()][1] != &NullPieceType96::nullPieceType96 )
			{
				promotionTypeCount++;
				promotions[1] = majorPieceTypes[piece->GetPlayerNumber()][1];
				if( majorPieceTypes[piece->GetPlayerNumber()][2] != &NullPieceType96::nullPieceType96 )
				{
					promotionTypeCount++;
					promotions[2] = majorPieceTypes[piece->GetPlayerNumber()][2];
					if( majorPieceTypes[piece->GetPlayerNumber()][3] != &NullPieceType96::nullPieceType96 )
					{
						promotionTypeCount++;
						promotions[3] = majorPieceTypes[piece->GetPlayerNumber()][3];
					}
				}
			}
		}
		return promotionTypeCount;
	}
	// *** STANDARD pawn promotion *** //
	return( Game::EnumeratePromotions( piece, fromSquare, toSquare, promotions, quiescentSearch ) );
}

void Game_12x8::AboutToStartThinking
	( int currentPlayer )
{
}

void Game_12x8::AboutToDeepenSearch
	( int currentPlayer )
{
}

void Game_12x8::DefaultSettings()
{
	Game::DefaultSettings();
}

void Game_12x8::SetPersonality
	( int personality )
{
	//	call base class implementation of SetPersonality
	Game::SetPersonality( personality );

	if( personality >= PERSONALITY_C )
	{
		piecesHaveChangingValues = false;
		phases[0].SetPawnDeficiencyFactor( 12 );
		phases[1].SetPawnDeficiencyFactor( 14 );
		phases[2].SetPawnDeficiencyFactor( 20 );
	}
	else
	{
		piecesHaveChangingValues = true;
		phases[0].SetPawnDeficiencyFactor( 10 );
		phases[1].SetPawnDeficiencyFactor( 12 );
		phases[2].SetPawnDeficiencyFactor( 16 );
	}
}
