
/***************************************************************************

                                 ChessV

                   COPYRIGHT (C) 2005 BY GREGORY STRONG

This file is part of ChessV.  ChessV is free software; you can redistribute
it and/or modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of the License, 
or (at your option) any later version.

ChessV is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
more details; the file 'COPYING' contains the License text, but if for
some reason you need a copy, please write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/


#include "StdAfx.h"
#include "../../ChessV.h"
#include "../../GameParameters.h"
#include "CWDAGame.h"


// *** DEBUG MEMORY ALLOCATION *** //
#ifdef _DEBUG
#define new SAFE_NEW
#endif


char defaultArmy[] = "Fabulous FIDEs";


void StartCWDAGame()
{
	int rtn = 0;
	if( !justLoaded )
	{
		rtn = (int) ::DialogBox( theInstance, MAKEINTRESOURCE(IDD_CWDA_SELECT_DIALOG), NULL, 
			reinterpret_cast<DLGPROC>(CWDA_SelectDlgProc) );
		if( rtn == CHESS_GAME_ALMOST_CHESS )
		{
			//	player selected Almost Chess army for both armies, therefore he is really
			//	playing Almost Chess, not Chess with Different Armies
			for( int i = 0; true; i++ )
			{
				GameInformation &game = supportedVariants[i];
				if( game.name == NULL )
					//	no more entries in supportedVariants array
					break;
				if( !strcmp( game.name, "Almost Chess" ) )
				{
					selectedVariant = &(supportedVariants[i]);
				}
			}
			
		}
		if( rtn == CHESS_GAME_ORTHODOX )
		{
			//	player selected Fabulous FIDEs army for both armies, therefore he is really
			//	playing Orthodox Chess, not Chess with Different Armies
			for( int i = 0; true; i++ )
			{
				GameInformation &game = supportedVariants[i];
				if( game.name == NULL )
					//	no more entries in supportedVariants array
					break;
				if( !strcmp( game.name, "Orthodox Chess" ) )
				{
					selectedVariant = &(supportedVariants[i]);
				}
			}
			
		}
	}

	theBoard = new Board( 8, 8 );
	char *player0Name = "White";
	char *player1Name = "Black";
	LookupStringParameter( "player1", player0Name );
	LookupStringParameter( "player2", player1Name );
	CreatePlayers( player0Name, player1Name, whiteComp, blackComp );
}

Game *CreateCWDAGame( Board &brd, Player &plr0, Player &plr1 )
{
	return new CWDAGame( brd, plr0, plr1 );
}

CWDAGame::CWDAGame( Board &board, Player &whitePlayer, Player &blackPlayer ):
	ChessGame(board, whitePlayer, blackPlayer)
{
	// *** OUTPOSTS *** //

	//	we use the user variable 'gameInt1' in the PieceType class
	//	to store the 'outpost factor' for a given piece.  this value will
	//	be multiplied by the square-values stored in the 'outpost' array.
	Phoenix::phoenix.gameInt1 = 6;
	WarElephant::warElephant.gameInt1 = 9;
	ChargingKnight::chargingKnight.gameInt1 = 10;
	NarrowKnight::narrowKnight.gameInt1 = 8;
	Bowman::bowman.gameInt1 = 5;
	Lion::lion.gameInt1 = 9;
	Knishop::knishop.gameInt1 = 9;
	Bishight::bishight.gameInt1 = 9;
	B4nD::b4nD.gameInt1 = 10;
	SV7::sv7.gameInt1 = 9;
	Cyborg::cyborg.gameInt1 = 9;
	Benz::benz.gameInt1 = 7;
	General::general.gameInt1 = 5;
}

void CWDAGame::AddPlayerPieceTypes( char *gameName )
{
	minorPieceTypes[0][2] = &NullPieceType64::nullPieceType64;
	minorPieceTypes[1][2] = &NullPieceType64::nullPieceType64;
	minorPieceTypes[0][3] = &NullPieceType64::nullPieceType64;
	minorPieceTypes[1][3] = &NullPieceType64::nullPieceType64;
	colorboundPieceTypes[0][0] = &NullPieceType64::nullPieceType64;
	colorboundPieceTypes[1][0] = &NullPieceType64::nullPieceType64;
	colorboundPieceTypes[0][1] = &NullPieceType64::nullPieceType64;
	colorboundPieceTypes[1][1] = &NullPieceType64::nullPieceType64;
	pawnPieceTypes[0] = &OrthodoxPawn::orthodoxPawn;
	pawnPieceTypes[1] = &OrthodoxPawn::orthodoxPawn;

	//	lookup White's army
	char *armyName;
	if( !LookupStringParameter( "white-army", armyName ) )
	{
		if( !strcmp( selectedVariant->name, "Almost Chess" ) )
			armyName = "Almost Chess";
		else
			armyName = "Fabulous FIDEs";
	}
	if( !strcmp( armyName, "Fabulous FIDEs" ) )
		whiteArmy = CWDA_ARMY_FABULOUS_FIDES;
	else if( !strcmp( armyName, "Colorbound Clobberers" ) )
		whiteArmy = CWDA_ARMY_COLORBOUND_CLOBBERERS;
	else if( !strcmp( armyName, "Colorbound Clobberers II" ) )
		whiteArmy = CWDA_ARMY_COLORBOUND_CLOBBERERS2;
	else if( !strcmp( armyName, "Nutty Knights" ) )
		whiteArmy = CWDA_ARMY_NUTTY_KNIGHTS;
	else if( !strcmp( armyName, "Remarkable Rookies" ) )
		whiteArmy = CWDA_ARMY_REMARKABLE_ROOKIES;
	else if( !strcmp( armyName, "Forward FIDEs" ) )
		whiteArmy = CWDA_ARMY_FORWARD_FIDES;
	else if( !strcmp( armyName, "All-Around Allstars" ) )
		whiteArmy = CWDA_ARMY_ALL_AROUND_ALLSTARS;
	else if( !strcmp( armyName, "Meticulous Mashers" ) )
		whiteArmy = CWDA_ARMY_METICULOUS_MASHERS;
	else if( !strcmp( armyName, "Almost Chess" ) )
		whiteArmy = CWDA_ARMY_ALMOST_CHESS;
	else if( !strcmp( armyName, "Deadly Demolishers" ) )
		whiteArmy = CWDA_ARMY_DEADLY_DEMOLISHERS;
	else if( !strcmp( armyName, "Mighty Men" ) )
		whiteArmy = CWDA_ARMY_MIGHTY_MEN;

	//	lookup Black's army
	if( !LookupStringParameter( "black-army", armyName ) )
	{
		if( !strcmp( selectedVariant->name, "Almost Chess" ) )
			armyName = "Almost Chess";
		else
			armyName = "Fabulous FIDEs";
	}
	if( !strcmp( armyName, "Fabulous FIDEs" ) )
		blackArmy = CWDA_ARMY_FABULOUS_FIDES;
	else if( !strcmp( armyName, "Colorbound Clobberers" ) )
		blackArmy = CWDA_ARMY_COLORBOUND_CLOBBERERS;
	else if( !strcmp( armyName, "Colorbound Clobberers II" ) )
		blackArmy = CWDA_ARMY_COLORBOUND_CLOBBERERS2;
	else if( !strcmp( armyName, "Nutty Knights" ) )
		blackArmy = CWDA_ARMY_NUTTY_KNIGHTS;
	else if( !strcmp( armyName, "Remarkable Rookies" ) )
		blackArmy = CWDA_ARMY_REMARKABLE_ROOKIES;
	else if( !strcmp( armyName, "Forward FIDEs" ) )
		blackArmy = CWDA_ARMY_FORWARD_FIDES;
	else if( !strcmp( armyName, "All-Around Allstars" ) )
		blackArmy = CWDA_ARMY_ALL_AROUND_ALLSTARS;
	else if( !strcmp( armyName, "Meticulous Mashers" ) )
		blackArmy = CWDA_ARMY_METICULOUS_MASHERS;
	else if( !strcmp( armyName, "Almost Chess" ) )
		blackArmy = CWDA_ARMY_ALMOST_CHESS;
	else if( !strcmp( armyName, "Deadly Demolishers" ) )
		blackArmy = CWDA_ARMY_DEADLY_DEMOLISHERS;
	else if( !strcmp( armyName, "Mighty Men" ) )
		blackArmy = CWDA_ARMY_MIGHTY_MEN;

	board.AddPlayerPieceTypeBothPlayers( OrthodoxKing::orthodoxKing );
	board.AddPlayerPieceTypeBothPlayers( OrthodoxPawn::orthodoxPawn );
	castlingPiecesAreColorbound[0] = false;
	castlingPiecesAreColorbound[1] = false;

	switch( blackArmy )
	{
	  case CWDA_ARMY_FABULOUS_FIDES:
		board.AddPlayerPieceTypeBothPlayers( OrthodoxRook::orthodoxRook );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxBishop::orthodoxBishop );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxKnight::orthodoxKnight );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxQueen::orthodoxQueen );
		castlingPieceTypes[1] = &OrthodoxRook::orthodoxRook;
		minorPieceTypes[1][0] = &OrthodoxKnight::orthodoxKnight;
		minorPieceTypes[1][1] = &OrthodoxBishop::orthodoxBishop;
		colorboundPieceTypes[1][0] = &OrthodoxBishop::orthodoxBishop;
		queenPieceTypes[1] = &OrthodoxQueen::orthodoxQueen;
		castlingPiecesSlideNorth[1] = true;
		break;
		
	  case CWDA_ARMY_COLORBOUND_CLOBBERERS:
		board.AddPlayerPieceTypeBothPlayers( Cleric::cleric );
		board.AddPlayerPieceTypeBothPlayers( Phoenix::phoenix );
		board.AddPlayerPieceTypeBothPlayers( WarElephant::warElephant );
		board.AddPlayerPieceTypeBothPlayers( Archbishop::archbishop );
		castlingPieceTypes[1] = &Cleric::cleric;
		colorboundPieceTypes[1][0] = &Cleric::cleric;
		colorboundPieceTypes[1][1] = &WarElephant::warElephant;
		minorPieceTypes[1][0] = &Phoenix::phoenix;
		minorPieceTypes[1][1] = &WarElephant::warElephant;
		queenPieceTypes[1] = &Archbishop::archbishop;
		castlingPiecesSlideNorth[1] = false;
		castlingPiecesAreColorbound[1] = true;
		break;

	  case CWDA_ARMY_COLORBOUND_CLOBBERERS2:
		board.AddPlayerPieceTypeBothPlayers( Cleric::cleric );
		board.AddPlayerPieceTypeBothPlayers( Phoenix::phoenix );
		board.AddPlayerPieceTypeBothPlayers( WarElephant::warElephant );
		board.AddPlayerPieceTypeBothPlayers( Archbishop::archbishop );
		castlingPieceTypes[1] = &Cleric::cleric;
		colorboundPieceTypes[1][0] = &Cleric::cleric;
		colorboundPieceTypes[1][1] = &WarElephant::warElephant;
		minorPieceTypes[1][0] = &Phoenix::phoenix;
		minorPieceTypes[1][1] = &WarElephant::warElephant;
		queenPieceTypes[1] = &Archbishop::archbishop;
		castlingPiecesSlideNorth[1] = false;
		castlingPiecesAreColorbound[1] = true;
		break;

	  case CWDA_ARMY_NUTTY_KNIGHTS:
		board.AddPlayerPieceTypeBothPlayers( ChargingRook::chargingRook );
		board.AddPlayerPieceTypeBothPlayers( NarrowKnight::narrowKnight );
		board.AddPlayerPieceTypeBothPlayers( ChargingKnight::chargingKnight );
		board.AddPlayerPieceTypeBothPlayers( Colonel::colonel );
		castlingPieceTypes[1] = &ChargingRook::chargingRook;
		minorPieceTypes[1][0] = &NarrowKnight::narrowKnight;
		minorPieceTypes[1][1] = &ChargingKnight::chargingKnight;
		queenPieceTypes[1] = &Colonel::colonel;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_REMARKABLE_ROOKIES:
		board.AddPlayerPieceTypeBothPlayers( ShortRook::shortRook );
		board.AddPlayerPieceTypeBothPlayers( Bowman::bowman );
		board.AddPlayerPieceTypeBothPlayers( Lion::lion );
		board.AddPlayerPieceTypeBothPlayers( Chancellor::chancellor );
		castlingPieceTypes[1] = &ShortRook::shortRook;
		minorPieceTypes[1][0] = &Bowman::bowman;
		minorPieceTypes[1][1] = &Lion::lion;
		queenPieceTypes[1] = &Chancellor::chancellor;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_FORWARD_FIDES:
		board.AddPlayerPieceTypeBothPlayers( ChargingRook::chargingRook );
		board.AddPlayerPieceTypeBothPlayers( Knishop::knishop );
		board.AddPlayerPieceTypeBothPlayers( Bishight::bishight );
		board.AddPlayerPieceTypeBothPlayers( Forequeen::forequeen );
		castlingPieceTypes[1] = &ChargingRook::chargingRook;
		minorPieceTypes[1][0] = &Knishop::knishop;
		minorPieceTypes[1][1] = &Bishight::bishight;
		queenPieceTypes[1] = &Forequeen::forequeen;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_ALL_AROUND_ALLSTARS:
		board.AddPlayerPieceTypeBothPlayers( ChargingRook::chargingRook );
		board.AddPlayerPieceTypeBothPlayers( Phoenix::phoenix );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxBishop::orthodoxBishop );
		board.AddPlayerPieceTypeBothPlayers( Chancellor::chancellor );
		castlingPieceTypes[1] = &ChargingRook::chargingRook;
		minorPieceTypes[1][0] = &Phoenix::phoenix;
		minorPieceTypes[1][1] = &OrthodoxBishop::orthodoxBishop;
		colorboundPieceTypes[1][0] = &OrthodoxBishop::orthodoxBishop;
		queenPieceTypes[1] = &Chancellor::chancellor;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_METICULOUS_MASHERS:
		board.AddPlayerPieceTypeBothPlayers( Fourfer::fourfer );
		board.AddPlayerPieceTypeBothPlayers( NarrowKnight::narrowKnight );
		board.AddPlayerPieceTypeBothPlayers( B4nD::b4nD );
		board.AddPlayerPieceTypeBothPlayers( N2R4::n2R4 );
		castlingPieceTypes[1] = &Fourfer::fourfer;
		minorPieceTypes[1][0] = &NarrowKnight::narrowKnight;
		minorPieceTypes[1][1] = &B4nD::b4nD;
		colorboundPieceTypes[1][0] = &B4nD::b4nD;
		queenPieceTypes[1] = &N2R4::n2R4;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_ALMOST_CHESS:
		board.AddPlayerPieceTypeBothPlayers( OrthodoxRook::orthodoxRook );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxBishop::orthodoxBishop );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxKnight::orthodoxKnight );
		board.AddPlayerPieceTypeBothPlayers( Chancellor::chancellor );
		castlingPieceTypes[1] = &OrthodoxRook::orthodoxRook;
		minorPieceTypes[1][0] = &OrthodoxKnight::orthodoxKnight;
		minorPieceTypes[1][1] = &OrthodoxBishop::orthodoxBishop;
		colorboundPieceTypes[1][0] = &OrthodoxBishop::orthodoxBishop;
		queenPieceTypes[1] = &Chancellor::chancellor;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_DEADLY_DEMOLISHERS:
		board.AddPlayerPieceTypeBothPlayers( Tiger::tiger );
		board.AddPlayerPieceTypeBothPlayers( Scout::scout );
		board.AddPlayerPieceTypeBothPlayers( Templar::templar );
		board.AddPlayerPieceTypeBothPlayers( Unicorn::unicorn );
		castlingPieceTypes[1] = &Tiger::tiger;
		minorPieceTypes[1][0] = &Scout::scout;
		minorPieceTypes[1][1] = &Templar::templar;
		colorboundPieceTypes[1][0] = &Templar::templar;
		queenPieceTypes[1] = &Unicorn::unicorn;
		castlingPiecesSlideNorth[1] = true;
		break;

	  case CWDA_ARMY_MIGHTY_MEN:
		board.AddPlayerPieceTypeBothPlayers( Pioneer::pioneer );
		board.AddPlayerPieceTypeBothPlayers( General::general );
		board.AddPlayerPieceTypeBothPlayers( Benz::benz );
		board.AddPlayerPieceTypeBothPlayers( Cyborg::cyborg );
		castlingPieceTypes[1] = &Pioneer::pioneer;
		minorPieceTypes[1][0] = &General::general;
		minorPieceTypes[1][1] = &Benz::benz;
		queenPieceTypes[1] = &Cyborg::cyborg;
		castlingPiecesSlideNorth[1] = true;
		break;
	}

	switch( whiteArmy )
	{
	  case CWDA_ARMY_FABULOUS_FIDES:
		board.AddPlayerPieceTypeBothPlayers( OrthodoxRook::orthodoxRook );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxBishop::orthodoxBishop );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxKnight::orthodoxKnight );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxQueen::orthodoxQueen );
		castlingPieceTypes[0] = &OrthodoxRook::orthodoxRook;
		minorPieceTypes[0][0] = &OrthodoxKnight::orthodoxKnight;
		minorPieceTypes[0][1] = &OrthodoxBishop::orthodoxBishop;
		colorboundPieceTypes[0][0] = &OrthodoxBishop::orthodoxBishop;
		queenPieceTypes[0] = &OrthodoxQueen::orthodoxQueen;
		castlingPiecesSlideNorth[0] = true;
		break;
		
	  case CWDA_ARMY_COLORBOUND_CLOBBERERS:
		board.AddPlayerPieceTypeBothPlayers( Cleric::cleric );
		board.AddPlayerPieceTypeBothPlayers( Phoenix::phoenix );
		board.AddPlayerPieceTypeBothPlayers( WarElephant::warElephant );
		board.AddPlayerPieceTypeBothPlayers( Archbishop::archbishop );
		castlingPieceTypes[0] = &Cleric::cleric;
		colorboundPieceTypes[0][0] = &Cleric::cleric;
		colorboundPieceTypes[0][1] = &WarElephant::warElephant;
		minorPieceTypes[0][0] = &Phoenix::phoenix;
		minorPieceTypes[0][1] = &WarElephant::warElephant;
		queenPieceTypes[0] = &Archbishop::archbishop;
		castlingPiecesSlideNorth[0] = false;
		castlingPiecesAreColorbound[0] = true;
		break;

	  case CWDA_ARMY_COLORBOUND_CLOBBERERS2:
		board.AddPlayerPieceTypeBothPlayers( Cleric::cleric );
		board.AddPlayerPieceTypeBothPlayers( Phoenix::phoenix );
		board.AddPlayerPieceTypeBothPlayers( WarElephant::warElephant );
		board.AddPlayerPieceTypeBothPlayers( Archbishop::archbishop );
		castlingPieceTypes[0] = &Cleric::cleric;
		colorboundPieceTypes[0][0] = &Cleric::cleric;
		colorboundPieceTypes[0][1] = &WarElephant::warElephant;
		minorPieceTypes[0][0] = &Phoenix::phoenix;
		minorPieceTypes[0][1] = &WarElephant::warElephant;
		queenPieceTypes[0] = &Archbishop::archbishop;
		castlingPiecesSlideNorth[0] = false;
		castlingPiecesAreColorbound[0] = true;
		break;

	  case CWDA_ARMY_NUTTY_KNIGHTS:
		board.AddPlayerPieceTypeBothPlayers( ChargingRook::chargingRook );
		board.AddPlayerPieceTypeBothPlayers( NarrowKnight::narrowKnight );
		board.AddPlayerPieceTypeBothPlayers( ChargingKnight::chargingKnight );
		board.AddPlayerPieceTypeBothPlayers( Colonel::colonel );
		castlingPieceTypes[0] = &ChargingRook::chargingRook;
		minorPieceTypes[0][0] = &NarrowKnight::narrowKnight;
		minorPieceTypes[0][1] = &ChargingKnight::chargingKnight;
		queenPieceTypes[0] = &Colonel::colonel;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_REMARKABLE_ROOKIES:
		board.AddPlayerPieceTypeBothPlayers( ShortRook::shortRook );
		board.AddPlayerPieceTypeBothPlayers( Bowman::bowman );
		board.AddPlayerPieceTypeBothPlayers( Lion::lion );
		board.AddPlayerPieceTypeBothPlayers( Chancellor::chancellor );
		castlingPieceTypes[0] = &ShortRook::shortRook;
		minorPieceTypes[0][0] = &Bowman::bowman;
		minorPieceTypes[0][1] = &Lion::lion;
		queenPieceTypes[0] = &Chancellor::chancellor;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_FORWARD_FIDES:
		board.AddPlayerPieceTypeBothPlayers( ChargingRook::chargingRook );
		board.AddPlayerPieceTypeBothPlayers( Knishop::knishop );
		board.AddPlayerPieceTypeBothPlayers( Bishight::bishight );
		board.AddPlayerPieceTypeBothPlayers( Forequeen::forequeen );
		castlingPieceTypes[0] = &ChargingRook::chargingRook;
		minorPieceTypes[0][0] = &Knishop::knishop;
		minorPieceTypes[0][1] = &Bishight::bishight;
		queenPieceTypes[0] = &Forequeen::forequeen;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_ALL_AROUND_ALLSTARS:
		board.AddPlayerPieceTypeBothPlayers( ChargingRook::chargingRook );
		board.AddPlayerPieceTypeBothPlayers( Phoenix::phoenix );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxBishop::orthodoxBishop );
		board.AddPlayerPieceTypeBothPlayers( Chancellor::chancellor );
		castlingPieceTypes[0] = &ChargingRook::chargingRook;
		minorPieceTypes[0][0] = &Phoenix::phoenix;
		minorPieceTypes[0][1] = &OrthodoxBishop::orthodoxBishop;
		colorboundPieceTypes[0][0] = &OrthodoxBishop::orthodoxBishop;
		queenPieceTypes[0] = &Chancellor::chancellor;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_METICULOUS_MASHERS:
		board.AddPlayerPieceTypeBothPlayers( Fourfer::fourfer );
		board.AddPlayerPieceTypeBothPlayers( Scout::scout );
		board.AddPlayerPieceTypeBothPlayers( B4nD::b4nD );
		board.AddPlayerPieceTypeBothPlayers( N2R4::n2R4 );
		castlingPieceTypes[0] = &Fourfer::fourfer;
		minorPieceTypes[0][0] = &NarrowKnight::narrowKnight;
		minorPieceTypes[0][1] = &B4nD::b4nD;
		colorboundPieceTypes[0][0] = &B4nD::b4nD;
		queenPieceTypes[0] = &N2R4::n2R4;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_ALMOST_CHESS:
		board.AddPlayerPieceTypeBothPlayers( OrthodoxRook::orthodoxRook );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxBishop::orthodoxBishop );
		board.AddPlayerPieceTypeBothPlayers( OrthodoxKnight::orthodoxKnight );
		board.AddPlayerPieceTypeBothPlayers( Chancellor::chancellor );
		castlingPieceTypes[0] = &OrthodoxRook::orthodoxRook;
		minorPieceTypes[0][0] = &OrthodoxKnight::orthodoxKnight;
		minorPieceTypes[0][1] = &OrthodoxBishop::orthodoxBishop;
		colorboundPieceTypes[0][0] = &OrthodoxBishop::orthodoxBishop;
		queenPieceTypes[0] = &Chancellor::chancellor;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_DEADLY_DEMOLISHERS:
		board.AddPlayerPieceTypeBothPlayers( Tiger::tiger );
		board.AddPlayerPieceTypeBothPlayers( Scout::scout );
		board.AddPlayerPieceTypeBothPlayers( Templar::templar );
		board.AddPlayerPieceTypeBothPlayers( Unicorn::unicorn );
		castlingPieceTypes[0] = &Tiger::tiger;
		minorPieceTypes[0][0] = &Scout::scout;
		minorPieceTypes[0][1] = &Templar::templar;
		colorboundPieceTypes[0][0] = &Templar::templar;
		queenPieceTypes[0] = &Unicorn::unicorn;
		castlingPiecesSlideNorth[0] = true;
		break;

	  case CWDA_ARMY_MIGHTY_MEN:
		board.AddPlayerPieceTypeBothPlayers( Pioneer::pioneer );
		board.AddPlayerPieceTypeBothPlayers( General::general );
		board.AddPlayerPieceTypeBothPlayers( Benz::benz );
		board.AddPlayerPieceTypeBothPlayers( Cyborg::cyborg );
		castlingPieceTypes[0] = &Pioneer::pioneer;
		minorPieceTypes[0][0] = &General::general;
		minorPieceTypes[0][1] = &Benz::benz;
		queenPieceTypes[0] = &Cyborg::cyborg;
		castlingPiecesSlideNorth[0] = true;
		break;
	}
}

char static_array[100];
char filename[30];

void CWDAGame::ChangeRulesByVariant
	( char *gameName,
	  char *&array,
	  char *&book )
{
	//	special case for Fabulous FIDEs vs. Fabulous FIDEs (Orthodox Chess)
	if( whiteArmy == CWDA_ARMY_FABULOUS_FIDES && 
		blackArmy == CWDA_ARMY_FABULOUS_FIDES )
	{
		//	place pieces
		array = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR";
		//	set name of opening book
		book = "openings\\OrthodoxChess.txt";
		//	activate Bit Boards 
		customMoveGeneration = true;
		customCaptureGeneration = true;
		//	all done!
		return;
	}

	//	piece together the filename of the opening book to use
	strcpy( filename, "openings\\CWDA_FaF_v_FaF.txt" );

	//	piece together starting array
	int array_cursor = 0;

	switch( blackArmy )
	{
	  case CWDA_ARMY_FABULOUS_FIDES:
		sprintf( static_array, "rnbqkbnr" );
		array_cursor = (int) strlen( static_array );
		break;
		
	  case CWDA_ARMY_COLORBOUND_CLOBBERERS:
		sprintf( static_array, "_cl_ph_weak_we_ph_cl" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'C';
		filename[21] = 'b';
		filename[22] = 'C';
		break;

	  case CWDA_ARMY_COLORBOUND_CLOBBERERS2:
		sprintf( static_array, "_cl_we_phak_ph_we_cl" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'C';
		filename[21] = 'C';
		filename[22] = '2';
		break;

	  case CWDA_ARMY_NUTTY_KNIGHTS:
		sprintf( static_array, "_cr_nn_cn_cok_cn_nn_cr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'N';
		filename[21] = 'u';
		filename[22] = 'K';
		break;

	  case CWDA_ARMY_REMARKABLE_ROOKIES:
		sprintf( static_array, "_sr_bolckl_bo_sr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'R';
		filename[21] = 'm';
		filename[22] = 'R';
		break;

	  case CWDA_ARMY_FORWARD_FIDES:
		sprintf( static_array, "_cr_nb_bn_fqk_bn_nb_cr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'F';
		filename[21] = 'w';
		filename[22] = 'F';
		break;

	  case CWDA_ARMY_ALL_AROUND_ALLSTARS:
		sprintf( static_array, "_cr_phbckb_ph_cr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'A';
		filename[21] = 'a';
		filename[22] = 'A';
		break;

	  case CWDA_ARMY_METICULOUS_MASHERS:
		sprintf( static_array, "_fr_nn_b4_n2k_b4_nn_fr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'M';
		filename[21] = 'e';
		filename[22] = 'M';
		break;

	  case CWDA_ARMY_ALMOST_CHESS:
		sprintf( static_array, "rnbckbnr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'A';
		filename[21] = 'l';
		filename[22] = 'C';
		break;

	  case CWDA_ARMY_DEADLY_DEMOLISHERS:
		sprintf( static_array, "d_sc_teuk_te_scd" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'D';
		filename[21] = 'e';
		filename[22] = 'D';
		break;

	  case CWDA_ARMY_MIGHTY_MEN:
		sprintf( static_array, "_prg_bz_cyk_bzg_pr" );
		array_cursor = (int) strlen( static_array );
		filename[20] = 'M';
		filename[21] = 'i';
		filename[22] = 'M';
		break;
	}

	sprintf( static_array + array_cursor, "/pppppppp/8/8/8/8/PPPPPPPP/" );
	array_cursor = (int) strlen( static_array );

	switch( whiteArmy )
	{
	  case CWDA_ARMY_FABULOUS_FIDES:
		sprintf( static_array + array_cursor, "RNBQKBNR" );
		break;
		
	  case CWDA_ARMY_COLORBOUND_CLOBBERERS:
		sprintf( static_array + array_cursor, "_CL_PH_WEAK_WE_PH_CL" );
		filename[14] = 'C';
		filename[15] = 'b';
		filename[16] = 'C';
		break;

	  case CWDA_ARMY_COLORBOUND_CLOBBERERS2:
		sprintf( static_array + array_cursor, "_BD_FA_WA_CAK_WA_FA_BD" );
		filename[14] = 'C';
		filename[15] = 'C';
		filename[16] = '2';
		break;

	  case CWDA_ARMY_NUTTY_KNIGHTS:
		sprintf( static_array + array_cursor, "_CR_NN_CN_COK_CN_NN_CR" );
		filename[14] = 'N';
		filename[15] = 'u';
		filename[16] = 'K';
		break;

	  case CWDA_ARMY_REMARKABLE_ROOKIES:
		sprintf( static_array + array_cursor, "_SR_BOLCKL_BO_SR" );
		filename[14] = 'R';
		filename[15] = 'm';
		filename[16] = 'R';
		break;

	  case CWDA_ARMY_FORWARD_FIDES:
		sprintf( static_array + array_cursor, "_CR_NB_BN_FQK_BN_NB_CR" );
		filename[14] = 'F';
		filename[15] = 'w';
		filename[16] = 'F';
		break;

	  case CWDA_ARMY_ALL_AROUND_ALLSTARS:
		sprintf( static_array + array_cursor, "_CR_PHBCKB_PH_CR" );
		filename[14] = 'A';
		filename[15] = 'a';
		filename[16] = 'A';
		break;

	  case CWDA_ARMY_METICULOUS_MASHERS:
		sprintf( static_array + array_cursor, "_FR_NN_B4_N2K_B4_NN_FR" );
		filename[14] = 'M';
		filename[15] = 'e';
		filename[16] = 'M';
		break;

	  case CWDA_ARMY_ALMOST_CHESS:
		sprintf( static_array + array_cursor, "RNBCKBNR" );
		filename[14] = 'A';
		filename[15] = 'l';
		filename[16] = 'C';
		break;

	  case CWDA_ARMY_DEADLY_DEMOLISHERS:
		sprintf( static_array + array_cursor, "D_SC_TEUK_TE_SCD" );
		filename[14] = 'D';
		filename[15] = 'e';
		filename[16] = 'D';
		break;

	  case CWDA_ARMY_MIGHTY_MEN:
		sprintf( static_array + array_cursor, "_PRG_BZ_CYK_BZG_PR" );
		filename[14] = 'M';
		filename[15] = 'i';
		filename[16] = 'M';
		break;
	}

	//	set starting array and opening book
	array = static_array;
	book = filename;

	// *** ARMY ADJUSTMENTS *** //

	// *** Fabulous FIDEs vs. Colorbound Clobberers *** //
	if( (whiteArmy == CWDA_ARMY_FABULOUS_FIDES &&
		 (blackArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS ||
		  blackArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS2)) ||
	    (blackArmy == CWDA_ARMY_FABULOUS_FIDES &&
		 (whiteArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS ||
		  whiteArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS2)) )
	{
		//	in this case, the rook is slightly more valuble than normal
		OrthodoxRook::orthodoxRook.SetArmyAdjustment( 500 );
		//	the queen is more valuble than normal (save it for the endgame!)
		OrthodoxQueen::orthodoxQueen.SetArmyAdjustment( 500 );
		//	the War Elephant (FAD) is worth a little less
		WarElephant::warElephant.SetArmyAdjustment( -100 );
		//	the Archbishop is worth a little more
		Archbishop::archbishop.SetArmyAdjustment( 50 );
	}
	// *** Fabulous FIDEs vs. Remarkable Rookies *** //
	if( (whiteArmy == CWDA_ARMY_FABULOUS_FIDES &&
		 blackArmy == CWDA_ARMY_REMARKABLE_ROOKIES) ||
	    (blackArmy == CWDA_ARMY_REMARKABLE_ROOKIES &&
		 whiteArmy == CWDA_ARMY_FABULOUS_FIDES) )
	{
		//	against the rookies, the Bishop is more valuble than normal
		OrthodoxBishop::orthodoxBishop.SetArmyAdjustment( 250 );
		//	the short rook is less valuble than usual against full rooks
		ShortRook::shortRook.SetArmyAdjustment( -200 );
		//	the Half-Duck is worth a little less
		Lion::lion.SetArmyAdjustment( -100 );
	}
	// *** Remarkable Rookies vs. Colorbound Clobberers *** //
	if( (whiteArmy == CWDA_ARMY_REMARKABLE_ROOKIES &&
		 (blackArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS ||
		  blackArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS2)) ||
	    (blackArmy == CWDA_ARMY_REMARKABLE_ROOKIES &&
		 (whiteArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS ||
		  whiteArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS2)) )
	{
		//	Half-duck is a tiny bit less valuable - equal to a FAD
		Lion::lion.SetArmyAdjustment( -50 );
		//	Short Rook is a bit more valuable, as the Clobberers have no rook-movers
		ShortRook::shortRook.SetArmyAdjustment( 200 );
	}
	// *** Nnutty Knights vs. Colorbound Clobberers *** //
	if( (whiteArmy == CWDA_ARMY_NUTTY_KNIGHTS &&
		 (blackArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS ||
		  blackArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS2)) ||
	    (blackArmy == CWDA_ARMY_NUTTY_KNIGHTS &&
		 (whiteArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS ||
		  whiteArmy == CWDA_ARMY_COLORBOUND_CLOBBERERS2)) )
	{
		//	Charging Rook is a bit more valuable, as the Clobberers have no rook-movers
		ChargingRook::chargingRook.SetArmyAdjustment( 250 );
	}
}

void CWDAGame::AddSpecialMoves
	( int currentPlayer, 
	  MovementList &stack, 
	  GameRec &gameRecord,
	  bool quiescentSearch )
{
	if( quiescentSearch || board.FastCheckTest() || castlingType != CASTLING_TYPE_STANDARD || 
		!castlingPiecesAreColorbound[currentPlayer] )
		//	pass this off to the standard ChessGame function
		ChessGame::AddSpecialMoves( currentPlayer, stack, gameRecord, quiescentSearch );
	else
	{
		//	check for en-passant
		if( enPassant )
			Game::AddEnPassantMoves( stack, gameRecord, quiescentSearch );
		
		if( board.GetCurrentPlayerNumber() == 0 )
		{
			//	white to move
			Piece *king = board.GetKing( 0 );
			if( !king->HasMoved() && king->GetSquareNumber() == 4 )
			{
				//	king side
				if( board.GetSquareContents( 7 ) != NULL &&
					!board.GetSquareContents( 7 )->HasMoved() )
				{
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 7 ), 6, 5 );
				}
				//	queen side
				if( board.GetSquareContents( 0 ) != NULL &&
					!board.GetSquareContents( 0 )->HasMoved() )
				{
					//	special adjustment for color-bound castling pieces
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 0 ), 1, 2 );
				}
			}
			else if( !king->HasMoved() && king->GetSquareNumber() == 3 )
			{
				//	king side
				if( board.GetSquareContents( 7 ) != NULL &&
					!board.GetSquareContents( 7 )->HasMoved() )
				{
					//	special adjustment for color-bound castling pieces
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 7 ), 6, 5 );
				}
				//	queen side
				if( board.GetSquareContents( 0 ) != NULL &&
					!board.GetSquareContents( 0 )->HasMoved() )
				{
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 0 ), 1, 2 );
				}
			}
		}
		else
		{
			//	black to move
			Piece *king = board.GetKing( 1 );
			if( !king->HasMoved() && king->GetSquareNumber() == 60 )
			{
				//	king side
				if( board.GetSquareContents( 63 ) != NULL &&
					!board.GetSquareContents( 63 )->HasMoved() )
				{
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 63 ), 62, 61 );
				}
				//	queen side
				if( board.GetSquareContents( 56 ) != NULL &&
					!board.GetSquareContents( 56 )->HasMoved() )
				{
					//	special adjustment for color-bound castling pieces
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 56 ), 57, 58 );
				}
			}
			else if( !king->HasMoved() && king->GetSquareNumber() == 59 )
			{
				//	king side
				if( board.GetSquareContents( 63 ) != NULL &&
					!board.GetSquareContents( 63 )->HasMoved() )
				{
					//	special adjustment for color-bound castling pieces
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 63 ), 62, 61 );
				}
				//	queen side
				if( board.GetSquareContents( 56 ) != NULL &&
					!board.GetSquareContents( 56 )->HasMoved() )
				{
					AddCastlingMove( stack, gameRecord, king, board.GetSquareContents( 56 ), 57, 58 );
				}
			}
		}
	}
}

void CWDAGame::GenerateMoves
	( int currentPlayer, 
	  MovementList &list )
{
	ChessGame::GenerateMoves( currentPlayer, list );
/*	BitBoard64 pieces;

	// *** PAWNS *** //
	pieces = OrthodoxPawn::orthodoxPawn.GetPieces( currentPlayer );
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		pieces.ClearBit( square );
		OrthodoxPawn::orthodoxPawn.BB64_GenerateDirectCaptures( list, currentPlayer, square );
		OrthodoxPawn::orthodoxPawn.BB64_GenerateDirectMovesNoCapture( list, currentPlayer, square );
	}
	//	handle the initial 2-step pawn move
	if( currentPlayer == 0 )
	{
		BitBoard64 secondRankPawns = player0PawnTwoSpaceZone & 
			OrthodoxPawn::orthodoxPawn.GetPieces( 0 );
		while( secondRankPawns )
		{
			int square = secondRankPawns.GetFirstBit();
			secondRankPawns.ClearBit( square );
			if( board.GetSquareContents( square + 8 ) == NULL && 
				board.GetSquareContents( square + 16 ) == NULL )
				//	squares are open; add the two-step move
				list.AddMove( square, square + 16 );
		}
	}
	else
	{
		BitBoard64 seventhRankPawns = player1PawnTwoSpaceZone & 
			OrthodoxPawn::orthodoxPawn.GetPieces( 1 );
		while( seventhRankPawns )
		{
			int square = seventhRankPawns.GetFirstBit();
			seventhRankPawns.ClearBit( square );
			if( board.GetSquareContents( square - 8 ) == NULL && 
				board.GetSquareContents( square - 16 ) == NULL )
				//	squares are open; add the two-step move
				list.AddMove( square, square - 16 );
		}
	}

	// *** DIRECT MOVERS *** //
	if( currentPlayer == 0 )
	{
		pieces = augmentedKnight0Type.GetPieces( currentPlayer );
		while( pieces )
		{
			int square = pieces.GetFirstBit();
			pieces.ClearBit( square );
			augmentedKnight0Type.BB64_GenerateDirectMoves( list, currentPlayer, square );
		}
	}
	else
	{
		pieces = augmentedKnight1Type.GetPieces( currentPlayer );
		while( pieces )
		{
			int square = pieces.GetFirstBit();
			pieces.ClearBit( square );
			augmentedKnight1Type.BB64_GenerateDirectMoves( list, currentPlayer, square );
		}
	}

	// *** KING *** //
	Piece *king = board.GetKing( currentPlayer );
	OrthodoxKing::orthodoxKing.BB64_GenerateDirectMoves( list, currentPlayer, king->GetSquareNumber() );

	// *** ROOK sliders *** //
	BitBoard64 friends = board.bb64_friends[currentPlayer];
	pieces = OrthodoxRook::orthodoxRook.GetPieces( currentPlayer ) |  
		OrthodoxQueen::orthodoxQueen.GetPieces( currentPlayer );
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		pieces.ToggleBit( square );

		BitBoard64 moves;
		moves = board.bb64_rook00attack[square * board.rook00attackWidth + 
			((board.bb64_blocker >> board.shift00[square]) & board.mask00[square])] | 
			board.bb64_rook90attack[square * board.rook90attackWidth + 
			((board.bb64_blocker90 >> board.shift90[square]) & board.mask90[square])];
		moves = moves & (~friends);
		if( moves )
			list.AddBitBoard64Moves( square, moves );
	}

	// *** BISHOP sliders *** // 
	pieces = OrthodoxBishop::orthodoxBishop.GetPieces( currentPlayer ) | 
		OrthodoxQueen::orthodoxQueen.GetPieces( currentPlayer );
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		pieces.ToggleBit( square );

		BitBoard64 moves;
		moves = board.bb64_bishop45attack[square * board.bishop45attackWidth + 
			((board.bb64_blocker45 >> board.shift45[square]) & board.mask45[square])] | 
			board.bb64_bishop135attack[square * board.bishop135attackWidth + 
			((board.bb64_blocker135 >> board.shift135[square]) & board.mask135[square])];
		moves = moves & (~friends);
		if( moves )
			list.AddBitBoard64Moves( square, moves );
	}

	//	check for en-passant
	AddSpecialMoves( currentPlayer, list, board.GetCurrentGameRecord(), false );*/
}

void CWDAGame::GenerateCaptures
	( int currentPlayer, 
	  MovementList &list )
{
	ChessGame::GenerateCaptures( currentPlayer, list );
/*	BitBoard64 pieces;

	// *** PAWNS *** //
	pieces = OrthodoxPawn::orthodoxPawn.GetPieces( currentPlayer );
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		pieces.ClearBit( square );
		OrthodoxPawn::orthodoxPawn.BB64_GenerateDirectCaptures( list, currentPlayer, square );
	}

	// *** DIRECT MOVERS *** //
	if( currentPlayer == 0 )
	{
		pieces = augmentedKnight0Type.GetPieces( currentPlayer );
		while( pieces )
		{
			int square = pieces.GetFirstBit();
			pieces.ClearBit( square );
			augmentedKnight0Type.BB64_GenerateDirectCaptures( list, currentPlayer, square );
		}
	}
	else
	{
		pieces = augmentedKnight1Type.GetPieces( currentPlayer );
		while( pieces )
		{
			int square = pieces.GetFirstBit();
			pieces.ClearBit( square );
			augmentedKnight1Type.BB64_GenerateDirectCaptures( list, currentPlayer, square );
		}
	}

	// *** KING *** //
	Piece *king = board.GetKing( currentPlayer );
	OrthodoxKing::orthodoxKing.BB64_GenerateDirectCaptures( list, currentPlayer, king->GetSquareNumber() );

	// *** ROOK sliders *** //
	BitBoard64 enemies = board.bb64_friends[FLIP(currentPlayer)];
	pieces = OrthodoxRook::orthodoxRook.GetPieces( currentPlayer ) |  
		OrthodoxQueen::orthodoxQueen.GetPieces( currentPlayer );
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		pieces.ToggleBit( square );

		BitBoard64 moves;
		moves = board.bb64_rook00attack[square * board.rook00attackWidth + 
			((board.bb64_blocker >> board.shift00[square]) & board.mask00[square])] | 
			board.bb64_rook90attack[square * board.rook90attackWidth + 
			((board.bb64_blocker90 >> board.shift90[square]) & board.mask90[square])];
		moves = moves & enemies;
		if( moves )
			list.AddBitBoard64Captures( square, moves );
	}

	// *** BISHOP sliders *** // 
	pieces = OrthodoxBishop::orthodoxBishop.GetPieces( currentPlayer ) | 
		OrthodoxQueen::orthodoxQueen.GetPieces( currentPlayer );
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		pieces.ToggleBit( square );

		BitBoard64 moves;
		moves = board.bb64_bishop45attack[square * board.bishop45attackWidth + 
			((board.bb64_blocker45 >> board.shift45[square]) & board.mask45[square])] | 
			board.bb64_bishop135attack[square * board.bishop135attackWidth + 
			((board.bb64_blocker135 >> board.shift135[square]) & board.mask135[square])];
		moves = moves & enemies;
		if( moves )
			list.AddBitBoard64Captures( square, moves );
	}

	//	check for en-passant
	if( enPassant )
		Game::AddEnPassantMoves( list, board.GetCurrentGameRecord(), true );*/
}

bool CWDAGame::IsSquareAttacked
	( int squareNumber, 
	  int playerNumber )
{
	BitBoard64 pieces;
	int otherPlayer = FLIP(playerNumber);

	// TODO: add bit-board support
	return board.IsSquareAttacked( squareNumber, playerNumber );
/*
	// *** PAWNS *** //
	if( OrthodoxPawn::orthodoxPawn.bb64_attack[otherPlayer][squareNumber] &
		OrthodoxPawn::orthodoxPawn.GetPieces( playerNumber ) )
		return true;

	// *** DIRECT MOVERS *** //
	if( playerNumber == 0 )
	{
		if( augmentedKnight0Type.bb64_attack[otherPlayer][squareNumber] & 
			augmentedKnight0Type.GetPieces( playerNumber ) )
			return true;
	}
	else
	{
		if( augmentedKnight1Type.bb64_attack[otherPlayer][squareNumber] & 
			augmentedKnight1Type.GetPieces( playerNumber ) )
			return true;
	}

	// *** KINGS *** //
	if( OrthodoxKing::orthodoxKing.bb64_attack[otherPlayer][squareNumber] &
		OrthodoxKing::orthodoxKing.GetPieces( playerNumber ) )
		return true;

	// *** ROOKS *** //
	pieces = OrthodoxRook::orthodoxRook.bb64_attack[otherPlayer][squareNumber] & 
		(OrthodoxRook::orthodoxRook.GetPieces( playerNumber ) | 
		 OrthodoxQueen::orthodoxQueen.GetPieces( playerNumber ));
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		if( ! ((bool) (board.bb64_rays[(squareNumber * board.GetNumberOfSquares()) + square] & 
			  board.bb64_blocker & ~BitBoard64::GetPositionBitBoard( square ))) )
			return true;
		pieces.ClearBit( square );
	}

	// *** BISHOPS *** //
	pieces = OrthodoxBishop::orthodoxBishop.bb64_attack[otherPlayer][squareNumber] & 
		(OrthodoxBishop::orthodoxBishop.GetPieces( playerNumber ) | 
		 OrthodoxQueen::orthodoxQueen.GetPieces( playerNumber ));
	while( pieces )
	{
		int square = pieces.GetFirstBit();
		if( ! ((bool) (board.bb64_rays[(squareNumber * board.GetNumberOfSquares()) + square] & 
			  board.bb64_blocker & ~BitBoard64::GetPositionBitBoard( square ))) )
			return true;
		pieces.ClearBit( square );
	}

	return false;*/
}

void CWDAGame::SetPersonality
	( int personality )
{
	//	call base class implementation of SetPersonality
	ChessGame::SetPersonality( personality );
}

